
"use client";

import { useState, useEffect } from "react";
import { showError, showSuccess } from "@/helpers/toastMessage";
import axios from "axios";

interface DomainsPageProps {
  showEditModal: boolean;
  setShowEditModal: (value: boolean) => void;
  onDomainUpdated: (updatedDomain: any) => void;
  currentDomainData: any | null;
}

const languages = ["en", "fr", "de", "it", "pt", "es"];

export default function DomainsPage({
  showEditModal,
  setShowEditModal,
  onDomainUpdated,
  currentDomainData,
}: DomainsPageProps) {
  const [domains, setDomains] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [formData, setFormData] = useState(currentDomainData);

  useEffect(() => {
    const fetchDomains = async () => {
      try {
        const { data } = await axios.get("/api/countries");
        setDomains(data);
      } catch (error) {
        console.error("Failed to fetch domains:", error);
        setDomains([]);
      } finally {
        setLoading(false);
      }
    };

    fetchDomains();
  }, []);

  useEffect(() => {
    setFormData(currentDomainData);
  }, [currentDomainData]);

  if (loading || !formData)
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-500"></div>
      </div>
    );

  const handleChange = (field: keyof any, value: any) => {
    setFormData((prev: any) => (prev ? { ...prev, [field]: value } : prev));
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!formData) return;

    try {
      const response = await fetch("/api/countries/update", {
        method: "PUT",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(formData),
      });

      if (!response.ok) throw new Error("Failed to update country");

      const updated = await response.json();
      showSuccess("Country updated successfully!");
      onDomainUpdated(updated);
      setShowEditModal(false);

      setDomains((prev) =>
        prev.map((d) => (d.id === updated.id ? updated : d))
      );
    } catch (err) {
      console.error(err);
      showError("Error updating country");
    }
  };

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4 z-50 modal-wrapper">
      <div className="bg-white rounded-lg max-w-3xl w-full">
        <div className="flex justify-between items-center px-6 py-4 border-b border-gray-200 max-md:px-4">
              <h2 className="text-xl font-semibold text-gray-900">Edit Domain</h2>
              <button onClick={() => setShowEditModal(false)} className="text-gray-500 hover:text-gray-700">
                  <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M6 18L18 6M6 6l12 12" />
                  </svg>
                </button>
            </div>

        <form className="" onSubmit={handleSubmit}>
          <div className="p-6 space-y-6 max-h-[70vh] overflow-y-auto">
            {/* Business Email */}
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Business Email
              </label>
              <input
                type="text"
                value={formData.businessEmail || ""}
                onChange={(e) => handleChange("businessEmail", e.target.value)}
                className="w-full border border-gray-300 rounded-lg px-3 py-2"
              />
            </div>

            {/* MAP IFRAME */}
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Map Iframe
              </label>
              <input
                type="text"
                value={formData.mapIframe || ""}
                onChange={(e) => handleChange("mapIframe", e.target.value)}
                className="w-full border border-gray-300 rounded-lg px-3 py-2"
              />
            </div>

            {/* 🔥 Multilingual Fields */}
            <div className="space-y-6">
              {languages.map((lang) => (
                <div
                  key={lang}
                  className="border border-gray-200 p-4 rounded-lg bg-gray-50"
                >
                  <h3 className="font-bold mb-4 uppercase text-blue-600">
                    {lang} Language Content
                  </h3>

                  {/* Location */}
                  <label className="block text-sm font-medium text-gray-700">
                    Location ({lang})
                  </label>
                  <input
                    type="text"
                    value={formData[`location_${lang}`] || ""}
                    onChange={(e) =>
                      handleChange(`location_${lang}` as any, e.target.value)
                    }
                    className="w-full border border-gray-300 rounded-lg px-3 py-2 mb-3"
                  />

                  {/* Sublocation */}
                  <label className="block text-sm font-medium text-gray-700">
                    Sub Location ({lang})
                  </label>
                  <input
                    type="text"
                    value={formData[`sublocation_${lang}`] || ""}
                    onChange={(e) =>
                      handleChange(`sublocation_${lang}` as any, e.target.value)
                    }
                    className="w-full border border-gray-300 rounded-lg px-3 py-2 mb-3"
                  />

                  {/* Description */}
                  <label className="block text-sm font-medium text-gray-700">
                    Short Description ({lang})
                  </label>
                  <textarea
                    value={formData[`shortDescription_${lang}`] || ""}
                    onChange={(e) =>
                      handleChange(
                        `shortDescription_${lang}` as any,
                        e.target.value
                      )
                    }
                    rows={3}
                    className="w-full border border-gray-300 rounded-lg px-3 py-2"
                  />
                </div>
              ))}
            </div>

            {/* Social Links */}
            <div className="grid grid-cols-2 gap-4 max-sm:grid-cols-1">
              <Input
                label="Facebook URL"
                field="facebookUrl"
                value={formData.facebookUrl}
                onChange={handleChange}
              />

              <Input
                label="Google URL"
                field="googleUrl"
                value={formData.googleUrl}
                onChange={handleChange}
              />

              <Input
                label="Instagram URL"
                field="instagramUrl"
                value={formData.instagramUrl}
                onChange={handleChange}
              />

              <Input
                label="LinkedIn URL"
                field="linkedinUrl"
                value={formData.linkedinUrl}
                onChange={handleChange}
              />

              <Input
                label="Twitter URL"
                field="twitterUrl"
                value={formData.twitterUrl}
                onChange={handleChange}
              />

              <Input
                label="Logo Header"
                field="logoHeader"
                value={formData.logoHeader}
                onChange={handleChange}
              />

              <Input
                label="Logo Footer"
                field="logoFooter"
                value={formData.logoFooter}
                onChange={handleChange}
              />

              <Input
                label="Support Email"
                field="supportEmail"
                value={formData.supportEmail}
                onChange={handleChange}
              />
              <Input
              label="Customer Phone"
              field="customerPhone"
              value={formData.customerPhone}
              onChange={handleChange}
            />
            </div>
          </div>

          <div className="p-4 px-6 flex justify-end space-x-4 border-t border-gray-200">
            <button
              type="button"
              className="px-4 py-2 border border-gray-300 rounded-lg"
              onClick={() => setShowEditModal(false)}
            >
              Cancel
            </button>
            <button
              type="submit"
              className="px-4 py-2 bg-blue-500 text-white rounded-lg"
            >
              Save Changes
            </button>
          </div>
        </form>
      </div>
    </div>
  );
}

function Input({ label, field, value, onChange }: any) {
  return (
    <div>
      <label className="block text-sm font-medium text-gray-700 mb-1">
        {label}
      </label>
      <input
        type="text"
        value={value || ""}
        onChange={(e) => onChange(field, e.target.value)}
        className="w-full border border-gray-300 rounded-lg px-3 py-2"
      />
    </div>
  );
}
