'use client';

import { useState, useEffect } from 'react';
import { Save, Upload, Facebook, Instagram, Linkedin, Twitter, Globe, Mail, Phone } from 'lucide-react';

interface SiteSetting {
  id: string;
  key: string;
  value: string | boolean | number;
  type: 'text' | 'json' | 'boolean' | 'number' | 'image';
  description?: string;
}

interface SocialMedia {
  facebook?: string;
  instagram?: string;
  linkedin?: string;
  twitter?: string;
}

const SettingField = ({ 
  setting, 
  value, 
  onChange 
}: { 
  setting: SiteSetting; 
  value: unknown;
  onChange: (key: string, value: unknown) => void;
}) => {
  const handleChange = (newValue: unknown) => {
    onChange(setting.key, newValue);
  };

  switch (setting.type) {
    case 'text':
      return (
        <input
          type="text"
          value={(value as string) || ''}
          onChange={(e) => handleChange(e.target.value)}
          className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent text-sm"
          placeholder={setting.description}
        />
      );
    
    case 'number':
      return (
        <input
          type="number"
          value={(value as string) || ''}
          onChange={(e) => handleChange(parseFloat(e.target.value) || 0)}
          className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent text-sm"
          placeholder={setting.description}
        />
      );
    
    case 'boolean':
      return (
        <label className="flex items-center">
          <input
            type="checkbox"
            checked={(value as boolean) || false}
            onChange={(e) => handleChange(e.target.checked)}
            className="mr-2"
          />
          <span className="text-sm text-gray-600">{setting.description}</span>
        </label>
      );
    
    case 'image':
      return (
        <div className="space-y-2">
          <div className="flex items-center space-x-4 max-sm:space-x-2">
            {value && typeof value === 'string' ? (
              <img
                src={value as string}
                alt="Preview"
                className="w-16 h-16 object-contain border border-gray-300 rounded"
              />
            ) : null}
            <div className="flex-1">
              <input
                type="url"
                value={(value as string) || ''}
                onChange={(e) => handleChange(e.target.value)}
                className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent text-sm"
                placeholder="Enter image URL"
              />
            </div>
            <button className="bg-gray-100 hover:bg-gray-200 text-gray-700 px-3 py-2 rounded-lg flex items-center space-x-2 cursor-pointer max-sm:space-x-0 max-sm:min-h-[38px] max-md:rounded-md">
              <Upload size={16} className="max-sm:ml-0" />
              <span className='max-sm:hidden'>Upload</span>
            </button>
          </div>
        </div>
      );
    
    default:
      return (
        <textarea
          value={(value as string) || ''}
          onChange={(e) => handleChange(e.target.value)}
          className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent text-sm"
          rows={3}
          placeholder={setting.description}
        />
      );
  }
};

const SocialMediaSection = ({ 
  socialMedia, 
  onChange 
}: { 
  socialMedia: SocialMedia;
  onChange: (social: SocialMedia) => void;
}) => {
  const socialPlatforms = [
    { key: 'facebook', label: 'Facebook', icon: Facebook, placeholder: 'https://facebook.com/yourpage' },
    { key: 'instagram', label: 'Instagram', icon: Instagram, placeholder: 'https://instagram.com/yourpage' },
    { key: 'linkedin', label: 'LinkedIn', icon: Linkedin, placeholder: 'https://linkedin.com/company/yourcompany' },
    { key: 'twitter', label: 'Twitter', icon: Twitter, placeholder: 'https://twitter.com/yourhandle' },
  ];

  return (
    <div className="space-y-4">
      {socialPlatforms.map((platform) => {
        const Icon = platform.icon;
        return (
          <div key={platform.key}>
            <label className="block text-sm font-medium text-gray-700 mb-2 max-md:mb-1">
              <div className="flex items-center space-x-2">
                <Icon size={16} />
                <span>{platform.label}</span>
              </div>
            </label>
            <input
              type="url"
              value={socialMedia[platform.key as keyof SocialMedia] || ''}
              onChange={(e) => onChange({
                ...socialMedia,
                [platform.key]: e.target.value
              })}
              className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent text-sm"
              placeholder={platform.placeholder}
            />
          </div>
        );
      })}
    </div>
  );
};

export default function SettingsPage() {
  const [settings, setSettings] = useState<Record<string, unknown>>({});
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [savedSuccess, setSavedSuccess] = useState(false);

  const settingsConfig: SiteSetting[] = [
    {
      id: '1',
      key: 'site_title',
      value: '',
      type: 'text',
      description: 'Main title of the website'
    },
    {
      id: '2',
      key: 'site_description',
      value: '',
      type: 'text',
      description: 'Short description of the website'
    },
    {
      id: '3',
      key: 'site_logo',
      value: '',
      type: 'image',
      description: 'Website logo'
    },
    {
      id: '4',
      key: 'contact_email',
      value: '',
      type: 'text',
      description: 'Contact email address'
    },
    {
      id: '5',
      key: 'contact_phone',
      value: '',
      type: 'text',
      description: 'Contact phone number'
    },
    {
      id: '6',
      key: 'maintenance_mode',
      value: false,
      type: 'boolean',
      description: 'Enable maintenance mode'
    },
    {
      id: '7',
      key: 'max_upload_size',
      value: 10,
      type: 'number',
      description: 'Maximum file upload size (MB)'
    }
  ];

  useEffect(() => {
    const fetchSettings = async () => {
      try {
        const response = await fetch('/api/settings');
        if (response.ok) {
          const data = await response.json();
          setSettings(data);
        } else {
          console.error('Failed to fetch settings');
          setSettings({});
        }
      } catch (error) {
        console.error('Error fetching settings:', error);
        setSettings({});
      } finally {
        setLoading(false);
      }
    };

    fetchSettings();
  }, []);

  const handleSettingChange = (key: string, value: unknown) => {
    setSettings(prev => ({
      ...prev,
      [key]: value
    }));
  };

  const handleSocialMediaChange = (socialMedia: SocialMedia) => {
    setSettings(prev => ({
      ...prev,
      social_media: socialMedia
    }));
  };

  const handleSave = async () => {
    setSaving(true);
    try {
      const response = await fetch('/api/settings', {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(settings),
      });

      if (response.ok) {
        setSavedSuccess(true);
        setTimeout(() => setSavedSuccess(false), 3000);
      } else {
        console.error('Failed to save settings');
      }
    } catch (error) {
      console.error('Failed to save settings:', error);
    } finally {
      setSaving(false);
    }
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-500"></div>
      </div>
    );
  }

  return (
    <div className="space-y-6 max-lg:space-y-4">
      <div className="flex justify-between items-start">
        <div className='max-md:pr-2'>
          <h1 className="text-2xl font-semibold text-gray-900 max-lg:text-xl">Site Settings</h1>
          <p className="text-gray-600 max-md:text-sm">Configure global site settings and preferences</p>
        </div>
        <button
          onClick={handleSave}
          disabled={saving}
          className={`px-4 py-2 rounded-lg flex items-center space-x-2 whitespace-nowrap text-white max-sm:text-sm ${
            savedSuccess 
              ? 'bg-green-500' 
              : saving 
                ? 'bg-gray-400' 
                : 'bg-blue-500 hover:bg-blue-600'
          }`}
        >
          <Save size={16} />
          <span>
            {savedSuccess ? 'Saved!' : saving ? 'Saving...' : 'Save Changes'}
          </span>
        </button>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        {/* General Settings */}
        <div className="bg-white rounded-lg shadow p-6 max-md:p-4">
          <h2 className="text-xl font-semibold text-gray-900 mb-4 max-md:text-lg max-md:mb-3">General Settings</h2>
          <div className="space-y-4">
            {settingsConfig.map((setting) => (
              <div key={setting.key}>
                <label className="block text-sm font-medium text-gray-700 mb-2 max-md:mb-1">
                  {setting.key.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase())}
                </label>
                <SettingField
                  setting={setting}
                  value={settings[setting.key]}
                  onChange={handleSettingChange}
                />
                {setting.description && setting.type !== 'boolean' && (
                  <p className="text-xs text-gray-500 mt-1">{setting.description}</p>
                )}
              </div>
            ))}
          </div>
        </div>

        {/* Social Media Settings */}
        <div className="bg-white rounded-lg shadow p-6 max-md:p-4">
          <h2 className="text-xl font-semibold text-gray-900 mb-4 max-md:text-lg max-md:mb-3">Social Media Links</h2>
          <SocialMediaSection
            socialMedia={settings.social_media || {}}
            onChange={handleSocialMediaChange}
          />
        </div>

        {/* Contact Information */}
        <div className="bg-white rounded-lg shadow p-6 max-md:p-4">
          <h2 className="text-xl font-semibold text-gray-900 mb-4 max-md:text-lg max-md:mb-3">Contact Information</h2>
          <div className="space-y-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2 max-md:mb-1">
                <div className="flex items-center space-x-2">
                  <Mail size={16} />
                  <span>Email Address</span>
                </div>
              </label>
              <input
                type="email"
                value={(settings.contact_email as string) || ''}
                onChange={(e) => handleSettingChange('contact_email', e.target.value)}
                className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent text-sm"
                placeholder="contact@example.com"
              />
            </div>
            
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2 max-md:mb-1">
                <div className="flex items-center space-x-2">
                  <Phone size={16} />
                  <span>Phone Number</span>
                </div>
              </label>
              <input
                type="tel"
                value={(settings.contact_phone as string) || ''}
                onChange={(e) => handleSettingChange('contact_phone', e.target.value)}
                className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent text-sm"
                placeholder="+41 123 456 789"
              />
            </div>
          </div>
        </div>

        {/* System Settings */}
        {/* <div className="bg-white rounded-lg shadow p-6">
          <h2 className="text-xl font-semibold text-gray-900 mb-6">System Settings</h2>
          <div className="space-y-4">
            <div>
              <label className="flex items-center">
                <input
                  type="checkbox"
                  checked={settings.maintenance_mode || false}
                  onChange={(e) => handleSettingChange('maintenance_mode', e.target.checked)}
                  className="mr-3"
                />
                <div>
                  <span className="text-sm font-medium text-gray-700">Maintenance Mode</span>
                  <p className="text-xs text-gray-500">Put the site in maintenance mode</p>
                </div>
              </label>
            </div>
            
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2 max-md:mb-1">
                Maximum Upload Size (MB)
              </label>
              <input
                type="number"
                value={settings.max_upload_size || 10}
                onChange={(e) => handleSettingChange('max_upload_size', parseInt(e.target.value) || 10)}
                className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                min="1"
                max="100"
              />
              <p className="text-xs text-gray-500 mt-1">Maximum file size for uploads</p>
            </div>
          </div>
        </div> */}
      </div>
    </div>
  );
}