'use client';

import { useState, useEffect } from 'react';
import { Search, Calendar, Eye, Edit, Trash2, Plus, DollarSign } from 'lucide-react';

interface Project {
  id: string;
  title: string;
  description: string;
  budget: number;
  duration: number;
  status: 'DRAFT' | 'OPEN' | 'IN_PROGRESS' | 'COMPLETED' | 'CANCELLED';
  client: {
    profile?: {
      firstName?: string;
      lastName?: string;
    };
    email: string;
  };
  category: {
    name: string;
  };
  country: {
    name: string;
    domain: string;
    currency: string;
    currencySymbol: string;
  };
  bids: Array<{
    id: string;
    status: 'PENDING' | 'ACCEPTED' | 'REJECTED';
  }>;
  publishedAt?: string;
  completedAt?: string;
  createdAt: string;
}

const FilterBar = ({ 
  searchTerm, 
  setSearchTerm, 
  statusFilter, 
  setStatusFilter,
  domainFilter,
  setDomainFilter,
}: {
  searchTerm: string;
  setSearchTerm: (term: string) => void;
  statusFilter: string;
  setStatusFilter: (status: string) => void;
  domainFilter: string;
  setDomainFilter: (domain: string) => void;
}) => (
  <div className="bg-white rounded-lg shadow p-4 mb-6">
    <div className="flex flex-wrap gap-4 items-center">
      <div className="flex-1 min-w-64">
        <div className="relative">
          <Search size={16} className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400" />
          <input
            type="text"
            placeholder="Search projects by title or description..."
            value={searchTerm}
            onChange={(e) => setSearchTerm(e.target.value)}
            className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
          />
        </div>
      </div>
      
      <select
        value={statusFilter}
        onChange={(e) => setStatusFilter(e.target.value)}
        className="border border-gray-300 rounded-lg px-3 py-2 text-sm min-h-[42px]"
      >
        <option value="">All Status</option>
        <option value="DRAFT">Draft</option>
        <option value="OPEN">Open</option>
        <option value="IN_PROGRESS">In Progress</option>
        <option value="COMPLETED">Completed</option>
        <option value="CANCELLED">Cancelled</option>
      </select>
      
      <select
        value={domainFilter}
        onChange={(e) => setDomainFilter(e.target.value)}
        className="border border-gray-300 rounded-lg px-3 py-2 text-sm min-h-[42px]"
      >
        <option value="">All Domains</option>
        <option value="smile24.ch">smile24.ch</option>
        <option value="smile24.fr">smile24.fr</option>
        <option value="smile24.de">smile24.de</option>
        <option value="smile24.it">smile24.it</option>
      </select>
      
      
    </div>
  </div>
);

const ProjectRow = ({ 
  project, 
  onViewProject, 
  onEditProject,
  onDeleteProject 
}: { 
  project: Project; 
  onViewProject: (projectId: string) => void;
  onEditProject: (projectId: string) => void;
  onDeleteProject: (projectId: string) => void;
}) => {
  const getStatusBadge = (status: string) => {
    const colors = {
      DRAFT: 'bg-gray-100 text-gray-800',
      OPEN: 'bg-blue-100 text-blue-800',
      IN_PROGRESS: 'bg-yellow-100 text-yellow-800',
      COMPLETED: 'bg-green-100 text-green-800',
      CANCELLED: 'bg-red-100 text-red-800',
    };
    return (
      <span className={`px-2 py-1 rounded-full text-xs font-medium ${colors[status as keyof typeof colors]}`}>
        {status.replace('_', ' ')}
      </span>
    );
  };

  const clientName = project.client.profile?.firstName && project.client.profile?.lastName
    ? `${project.client.profile.firstName} ${project.client.profile.lastName}`
    : project.client.email;

  const bidCount = project.bids.length;
  const acceptedBids = project.bids.filter(bid => bid.status === 'ACCEPTED').length;

  return (
    <tr className="hover:bg-gray-50">
      <td className="px-6 py-4">
        <div>
          <div className="text-sm font-medium text-gray-900 truncate max-w-xs">
            {project.title}
          </div>
          <div className="text-sm text-gray-500 truncate max-w-xs">
            {project.description}
          </div>
        </div>
      </td>
      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
        {clientName}
      </td>
      <td className="px-6 py-4 whitespace-nowrap">
        {getStatusBadge(project.status)}
      </td>
      {/* <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
        <div className="flex items-center">
          <DollarSign size={14} className="text-gray-400 mr-1" />
          {project.country.currencySymbol}{project.budget.toLocaleString()}
        </div>
      </td> */}
      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
        {project.category.name}
      </td>
      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
        {project.country.name}
      </td>
      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
        {bidCount} ({acceptedBids} accepted)
      </td>
      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
        {new Date(project.createdAt).toLocaleDateString()}
      </td>
      {/* <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
        <div className="flex items-center space-x-2">
          <button 
            onClick={() => onViewProject(project.id)}
            className="text-blue-600 hover:text-blue-900 p-1"
            title="View project details"
          >
            <Eye size={16} />
          </button>
        </div>
      </td> */}
    </tr>
  );
};

export default function ProjectsPage() {
  const [projects, setProjects] = useState<Project[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState('');
  const [domainFilter, setDomainFilter] = useState('');
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const [selectedProject, setSelectedProject] = useState<unknown>(null);
  const [showProjectModal, setShowProjectModal] = useState(false);
  const [showEditModal, setShowEditModal] = useState(false);
  const projectsPerPage = 10;

  useEffect(() => {
    const fetchProjects = async () => {
      try {
        const params = new URLSearchParams({
          page: currentPage.toString(),
          limit: projectsPerPage.toString()
        });
        
        if (statusFilter) params.set('status', statusFilter);
        if (domainFilter) params.set('domain', domainFilter);

        const response = await fetch(`/api/projects?${params}`);
        if (response.ok) {
          const data = await response.json();
          setProjects(data.projects);
          setTotalPages(data.totalPages);
        } else {
          console.error('Failed to fetch projects');
          setProjects([]);
        }
      } catch (error) {
        console.error('Error fetching projects:', error);
        setProjects([]);
      } finally {
        setLoading(false);
      }
    };

    fetchProjects();
  }, [domainFilter, statusFilter, currentPage]);

  const handleViewProject = async (projectId: string) => {
    try {
      const response = await fetch(`/api/projects/${projectId}`);
      if (response.ok) {
        const projectData = await response.json();
        setSelectedProject(projectData);
        setShowProjectModal(true);
      } else {
        console.error('Failed to fetch project details');
      }
    } catch (error) {
      console.error('Error fetching project details:', error);
    }
  };

  const handleEditProject = async (projectId: string) => {
    try {
      const response = await fetch(`/api/projects/${projectId}`);
      if (response.ok) {
        const projectData = await response.json();
        setSelectedProject(projectData);
        setShowEditModal(true);
      } else {
        console.error('Failed to fetch project details');
      }
    } catch (error) {
      console.error('Error fetching project details:', error);
    }
  };

  const handleDeleteProject = async (projectId: string) => {
    if (window.confirm('Are you sure you want to delete this project? This action cannot be undone.')) {
      try {
        const response = await fetch(`/api/projects/${projectId}`, {
          method: 'DELETE',
        });

        if (response.ok) {
          // Remove the project from the local state
          setProjects(prevProjects => 
            prevProjects.filter(project => project.id !== projectId)
          );
        } else {
          const errorData = await response.json();
          console.error('Failed to delete project:', errorData.error);
          alert('Failed to delete project: ' + errorData.error);
        }
      } catch (error) {
        console.error('Error deleting project:', error);
        alert('Error deleting project');
      }
    }
  };

  const filteredProjects = projects.filter(project => {
    const matchesSearch = !searchTerm || 
      project.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
      project.description.toLowerCase().includes(searchTerm.toLowerCase());
    
    const matchesStatus = !statusFilter || project.status === statusFilter;
    const matchesDomain = !domainFilter || project.country.domain === domainFilter;
    
    return matchesSearch && matchesStatus && matchesDomain;
  });

  const paginatedProjects = filteredProjects.slice(
    (currentPage - 1) * projectsPerPage,
    currentPage * projectsPerPage
  );

  // Calculate summary stats
  const totalBudget = filteredProjects.reduce((sum, project) => sum + project.budget, 0);
  const completedProjects = filteredProjects.filter(p => p.status === 'COMPLETED').length;
  const activeProjects = filteredProjects.filter(p => ['OPEN', 'IN_PROGRESS'].includes(p.status)).length;

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-500"></div>
      </div>
    );
  }

  return (
    <div className="space-y-6 max-lg:space-y-4">
      <div className="flex justify-between items-center">
        <div>
          <h1 className="text-2xl font-semibold text-gray-900 max-lg:text-xl">Projects</h1>
          {/* <p className="text-gray-600 mt-2">Manage all projects on the platform</p> */}
        </div>
        {/* <button className="bg-blue-500 hover:bg-blue-600 text-white px-4 py-2 rounded-lg flex items-center space-x-2">
          <Plus size={16} />
          <span>Add Project</span>
        </button> */}
      </div>

      {/* Summary Stats */}
      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        <div className="bg-white rounded-lg shadow p-4">
          <div className="flex items-center">
            <div className="flex-1">
              <p className="text-sm font-medium text-gray-600">Total Projects</p>
              <p className="text-2xl font-bold text-gray-900">{filteredProjects.length}</p>
            </div>
          </div>
        </div>
        <div className="bg-white rounded-lg shadow p-4">
          <div className="flex items-center">
            <div className="flex-1">
              <p className="text-sm font-medium text-gray-600">Active Projects</p>
              <p className="text-2xl font-bold text-blue-600">{activeProjects}</p>
            </div>
          </div>
        </div>
        {/* <div className="bg-white rounded-lg shadow p-6">
          <div className="flex items-center">
            <div className="flex-1">
              <p className="text-sm font-medium text-gray-600">Total Budget</p>
              <p className="text-2xl font-bold text-green-600">${totalBudget.toLocaleString()}</p>
            </div>
          </div>
        </div> */}
      </div>

      <FilterBar
        searchTerm={searchTerm}
        setSearchTerm={setSearchTerm}
        statusFilter={statusFilter}
        setStatusFilter={setStatusFilter}
        domainFilter={domainFilter}
        setDomainFilter={setDomainFilter}
      />

      <div className="bg-white rounded-lg shadow overflow-hidden">
        <div className="overflow-x-auto">
          <table className="min-w-full divide-y divide-gray-200">
            <thead className="bg-gray-50">
              <tr>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Project
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Client
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Status
                </th>
                {/* <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Budget
                </th> */}
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Category
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Country
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Bids
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Created
                </th>
                {/* <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Actions
                </th> */}
              </tr>
            </thead>
            <tbody className="bg-white divide-y divide-gray-200">
              {paginatedProjects.map((project) => (
                <ProjectRow 
                  key={project.id} 
                  project={project} 
                  onViewProject={handleViewProject}
                  onEditProject={handleEditProject}
                  onDeleteProject={handleDeleteProject}
                />
              ))}
            </tbody>
          </table>
        </div>
      </div>

      {/* Pagination */}
      <div className="flex items-center justify-between">
        <div className="text-sm text-gray-700">
          Showing {(currentPage - 1) * projectsPerPage + 1} to {Math.min(currentPage * projectsPerPage, filteredProjects.length)} of {filteredProjects.length} projects
        </div>
        <div className="flex space-x-2">
          <button
            onClick={() => setCurrentPage(Math.max(1, currentPage - 1))}
            disabled={currentPage === 1}
            className="px-3 py-2 border border-gray-300 rounded-lg text-sm font-medium text-gray-700 hover:bg-gray-50 disabled:opacity-50"
          >
            Previous
          </button>
          <button
            onClick={() => setCurrentPage(Math.min(totalPages, currentPage + 1))}
            disabled={currentPage === totalPages}
            className="px-3 py-2 border border-gray-300 rounded-lg text-sm font-medium text-gray-700 hover:bg-gray-50 disabled:opacity-50"
          >
            Next
          </button>
        </div>
      </div>
    </div>
  );
}