'use client';

import { useState, useEffect } from 'react';
import { Plus, Edit, Trash2, DollarSign, Calendar, Users } from 'lucide-react';

interface Package {
  id: string;
  name: string;
  description?: string;
  features: string[];
  price: number;
  duration: number;
  categoryLimit?: number;
  bidLimit?: number;
  isActive: boolean;
  country: {
    name: string;
    domain: string;
    currency: string;
    currencySymbol: string;
  };
  subscriptions: Array<{
    status: 'ACTIVE' | 'EXPIRED' | 'CANCELLED';
  }>;
  createdAt: string;
  updatedAt: string;
}

const PackageCard = ({ 
  package: pkg, 
  onEdit, 
  onDelete 
}: { 
  package: Package; 
  onEdit: (pkg: Package) => void;
  onDelete: (id: string) => void;
}) => {
  const activeSubscriptions = pkg.subscriptions.filter(sub => sub.status === 'ACTIVE').length;
  
  return (
    <div className="bg-white rounded-lg shadow-lg border border-gray-200 overflow-hidden">
      <div className="p-6">
        <div className="flex justify-between items-start mb-4">
          <div>
            <h3 className="text-xl font-bold text-gray-900">{pkg.name}</h3>
            <p className="text-sm text-gray-600">{pkg.country.name}</p>
          </div>
          <div className="flex items-center space-x-2">
            <span className={`px-2 py-1 rounded-full text-xs font-medium ${
              pkg.isActive 
                ? 'bg-green-100 text-green-800' 
                : 'bg-red-100 text-red-800'
            }`}>
              {pkg.isActive ? 'Active' : 'Inactive'}
            </span>
            <button
              onClick={() => onEdit(pkg)}
              className="text-blue-600 hover:text-blue-800 p-1"
            >
              <Edit size={16} />
            </button>
            <button
              onClick={() => onDelete(pkg.id)}
              className="text-red-600 hover:text-red-800 p-1"
            >
              <Trash2 size={16} />
            </button>
          </div>
        </div>

        <div className="mb-4">
          <div className="flex items-baseline">
            <span className="text-3xl font-bold text-gray-900 max-lg:text-2xl">
              {pkg.country.currencySymbol}{pkg.price}
            </span>
            <span className="text-gray-600 ml-2">/{pkg.duration} days</span>
          </div>
        </div>

        {pkg.description && (
          <p className="text-gray-600 mb-4">{pkg.description}</p>
        )}

        <div className="space-y-2 mb-4">
          {pkg.features.map((feature, index) => (
            <div key={index} className="flex items-center text-sm text-gray-600">
              <div className="w-2 h-2 bg-blue-500 rounded-full mr-3"></div>
              {feature}
            </div>
          ))}
        </div>

        <div className="grid grid-cols-2 gap-4 mb-4 pt-4 border-t border-gray-200">
          {pkg.categoryLimit && (
            <div className="text-center">
              <div className="text-2xl font-bold text-blue-600">{pkg.categoryLimit}</div>
              <div className="text-xs text-gray-600">Categories</div>
            </div>
          )}
          {pkg.bidLimit && (
            <div className="text-center">
              <div className="text-2xl font-bold text-green-600">{pkg.bidLimit}</div>
              <div className="text-xs text-gray-600">Bids/Month</div>
            </div>
          )}
        </div>

        <div className="flex items-center justify-between pt-4 border-t border-gray-200">
          <div className="flex items-center text-sm text-gray-600">
            <Users size={16} className="mr-1" />
            {activeSubscriptions} active users
          </div>
          <div className="text-xs text-gray-500">
            Updated {new Date(pkg.updatedAt).toLocaleDateString()}
          </div>
        </div>
      </div>
    </div>
  );
};

const EditPackageModal = ({ 
  package: pkg, 
  isOpen, 
  onClose, 
  onSave 
}: {
  package: Package | null;
  isOpen: boolean;
  onClose: () => void;
  onSave: (pkg: Package) => void;
}) => {
  const [formData, setFormData] = useState<Partial<Package>>({});

  useEffect(() => {
    if (pkg) {
      setFormData(pkg);
    }
  }, [pkg]);

  if (!isOpen || !pkg) return null;

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    onSave(formData as Package);
  };

  return (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center p-4 z-50">
      <div className="bg-white rounded-lg max-w-2xl w-full max-h-[90vh] overflow-y-auto">
        <div className="p-6 border-b border-gray-200">
          <h2 className="text-xl font-bold text-gray-900">Edit Package</h2>
        </div>
        
        <form onSubmit={handleSubmit} className="p-6 space-y-4">
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Package Name
            </label>
            <input
              type="text"
              value={formData.name || ''}
              onChange={(e) => setFormData({ ...formData, name: e.target.value })}
              className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              required
            />
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">
              Description
            </label>
            <textarea
              value={formData.description || ''}
              onChange={(e) => setFormData({ ...formData, description: e.target.value })}
              className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              rows={3}
            />
          </div>

          <div className="grid grid-cols-2 gap-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Price ({pkg.country.currencySymbol})
              </label>
              <input
                type="number"
                step="0.01"
                value={formData.price || ''}
                onChange={(e) => setFormData({ ...formData, price: parseFloat(e.target.value) })}
                className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                required
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Duration (days)
              </label>
              <input
                type="number"
                value={formData.duration || ''}
                onChange={(e) => setFormData({ ...formData, duration: parseInt(e.target.value) })}
                className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                required
              />
            </div>
          </div>

          <div className="grid grid-cols-2 gap-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Category Limit
              </label>
              <input
                type="number"
                value={formData.categoryLimit || ''}
                onChange={(e) => setFormData({ ...formData, categoryLimit: parseInt(e.target.value) || undefined })}
                className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              />
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Bid Limit (per month)
              </label>
              <input
                type="number"
                value={formData.bidLimit || ''}
                onChange={(e) => setFormData({ ...formData, bidLimit: parseInt(e.target.value) || undefined })}
                className="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              />
            </div>
          </div>

          <div>
            <label className="flex items-center">
              <input
                type="checkbox"
                checked={formData.isActive || false}
                onChange={(e) => setFormData({ ...formData, isActive: e.target.checked })}
                className="mr-2"
              />
              <span className="text-sm font-medium text-gray-700">Active</span>
            </label>
          </div>

          <div className="flex justify-end space-x-4 pt-4 border-t border-gray-200">
            <button
              type="button"
              onClick={onClose}
              className="px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50"
            >
              Cancel
            </button>
            <button
              type="submit"
              className="px-4 py-2 bg-blue-500 text-white rounded-lg hover:bg-blue-600"
            >
              Save Changes
            </button>
          </div>
        </form>
      </div>
    </div>
  );
};

export default function PackagesPage() {
  const [packages, setPackages] = useState<Package[]>([]);
  const [loading, setLoading] = useState(true);
  const [editingPackage, setEditingPackage] = useState<Package | null>(null);
  const [showEditModal, setShowEditModal] = useState(false);

  useEffect(() => {
    const fetchPackages = async () => {
      try {
        const response = await fetch('/api/packages');
        if (response.ok) {
          const data = await response.json();
          setPackages(data);
        } else {
          console.error('Failed to fetch packages');
          setPackages([]);
        }
      } catch (error) {
        console.error('Error fetching packages:', error);
        setPackages([]);
      } finally {
        setLoading(false);
      }
    };

    fetchPackages();
  }, []);

  const handleEditPackage = (pkg: Package) => {
    setEditingPackage(pkg);
    setShowEditModal(true);
  };

  const handleSavePackage = async (updatedPackage: Package) => {
    try {
      const response = await fetch('/api/packages', {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(updatedPackage),
      });

      if (response.ok) {
        const savedPackage = await response.json();
        setPackages(packages.map(pkg => 
          pkg.id === savedPackage.id ? savedPackage : pkg
        ));
        setShowEditModal(false);
        setEditingPackage(null);
      } else {
        console.error('Failed to update package');
      }
    } catch (error) {
      console.error('Error updating package:', error);
    }
  };

  const handleDeletePackage = (id: string) => {
    if (confirm('Are you sure you want to delete this package?')) {
      setPackages(packages.filter(pkg => pkg.id !== id));
    }
  };

  // Group packages by country
  const packagesByCountry = packages.reduce((acc, pkg) => {
    const country = pkg.country.name;
    if (!acc[country]) {
      acc[country] = [];
    }
    acc[country].push(pkg);
    return acc;
  }, {} as Record<string, Package[]>);

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-500"></div>
      </div>
    );
  }

  return (
    <div className="space-y-6 max-lg:space-y-4">
      <div className="flex justify-between items-center">
        <div>
          <h1 className="text-2xl font-semibold text-gray-900 max-lg:text-xl">Packages & Pricing</h1>
          <p className="text-gray-600 mt-2">Manage subscription packages for each domain</p>
        </div>
        {/* <button className="bg-blue-500 hover:bg-blue-600 text-white px-4 py-2 rounded-lg flex items-center space-x-2">
          <Plus size={16} />
          <span>Add Package</span>
        </button> */}
      </div>

      {Object.entries(packagesByCountry).map(([country, countryPackages]) => (
        <div key={country} className="space-y-4">
          <div className="flex items-center space-x-2">
            <h2 className="text-xl font-semibold text-gray-900">{country}</h2>
            <span className="text-sm text-gray-500">
              ({countryPackages[0]?.country.domain})
            </span>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
            {countryPackages.map((pkg) => (
              <PackageCard
                key={pkg.id}
                package={pkg}
                onEdit={handleEditPackage}
                onDelete={handleDeletePackage}
              />
            ))}
          </div>
        </div>
      ))}

      <EditPackageModal
        package={editingPackage}
        isOpen={showEditModal}
        onClose={() => {
          setShowEditModal(false);
          setEditingPackage(null);
        }}
        onSave={handleSavePackage}
      />
    </div>
  );
}