'use client';

import { useState, useEffect } from 'react';
import { Plus, Edit, Trash2, Check, X } from 'lucide-react';

interface Language {
  id: string;
  code: string;
  name: string;
  nativeName: string;
  isActive: boolean;
  createdAt: string;
}

export default function LanguagesPage() {
  const [languages, setLanguages] = useState<Language[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const fetchLanguages = async () => {
      try {
        const response = await fetch('/api/languages');
        if (response.ok) {
          const data = await response.json();
          setLanguages(data);
        } else {
          console.error('Failed to fetch languages');
          setLanguages([]);
        }
      } catch (error) {
        console.error('Error fetching languages:', error);
        setLanguages([]);
      } finally {
        setLoading(false);
      }
    };

    fetchLanguages();
  }, []);

  const toggleLanguageStatus = async (id: string) => {
    const language = languages.find(lang => lang.id === id);
    if (!language) return;

    try {
      const response = await fetch('/api/languages', {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          id,
          isActive: !language.isActive
        }),
      });

      if (response.ok) {
        setLanguages(languages.map(lang => 
          lang.id === id ? { ...lang, isActive: !lang.isActive } : lang
        ));
      } else {
        console.error('Failed to update language status');
      }
    } catch (error) {
      console.error('Error updating language status:', error);
    }
  };

  if (loading) {
    return <div className="flex items-center justify-center h-64">
      <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-500"></div>
    </div>;
  }

  const activeLanguages = languages.filter(lang => lang.isActive);
  const inactiveLanguages = languages.filter(lang => !lang.isActive);

  return (
    <div className="space-y-6 max-lg:space-y-4">
      <div className="flex justify-between items-center">
        <div>
          <h1 className="text-2xl font-semibold text-gray-900 max-lg:text-xl">Languages</h1>
          {/* <p className="text-gray-600 mt-2">Manage available languages for the platform</p> */}
        </div>
        {/* <button className="bg-blue-500 hover:bg-blue-600 text-white px-4 py-2 rounded-lg flex items-center space-x-2">
          <Plus size={16} />
          <span>Add Language</span>
        </button> */}
      </div>

      {/* Active Languages */}
      <div className="bg-white rounded-lg shadow">
        <div className="px-6 py-4 border-b border-gray-200 max-md:px-4">
          <h2 className="text-lg font-semibold text-gray-900">
            Active Languages ({activeLanguages.length})
          </h2>
          <p className="text-sm text-gray-600">These languages will be shown in frontend forms</p>
        </div>
        <div className="p-6 max-md:p-4">
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
            {activeLanguages.map((language) => (
              <div key={language.id} className="border border-green-200 bg-green-50 rounded-lg p-4">
                <div className="flex justify-between items-start mb-3">
                  <div>
                    <h3 className="font-semibold text-gray-900">{language.name}</h3>
                    <p className="text-sm text-gray-600">{language.nativeName}</p>
                  </div>
                  <div className="flex items-center space-x-2">
                    <span className="px-2 py-1 bg-green-100 text-green-800 rounded-full text-xs font-medium">
                      {language.code.toUpperCase()}
                    </span>
                  </div>
                </div>
                {/* <div className="flex justify-between items-center">
                  <span className="flex items-center text-sm text-green-600">
                    <Check size={16} className="mr-1" />
                    Active
                  </span>
                  <div className="flex space-x-2">
                    <button
                      onClick={() => toggleLanguageStatus(language.id)}
                      className="text-red-600 hover:text-red-800 text-sm"
                    >
                      Deactivate
                    </button>
                    <button className="text-blue-600 hover:text-blue-800 p-1">
                      <Edit size={14} />
                    </button>
                  </div>
                </div> */}
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* Inactive Languages */}
      {inactiveLanguages.length > 0 && (
        <div className="bg-white rounded-lg shadow">
          <div className="px-6 py-4 border-b border-gray-200 max-md:px-4">
            <h2 className="text-lg font-semibold text-gray-900">
              Inactive Languages ({inactiveLanguages.length})
            </h2>
            <p className="text-sm text-gray-600">These languages are not shown in the frontend</p>
          </div>
          <div className="p-6 max-md:p-4">
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
              {inactiveLanguages.map((language) => (
                <div key={language.id} className="border border-gray-200 bg-gray-50 rounded-lg p-4">
                  <div className="flex justify-between items-start mb-3">
                    <div>
                      <h3 className="font-semibold text-gray-900">{language.name}</h3>
                      <p className="text-sm text-gray-600">{language.nativeName}</p>
                    </div>
                    <div className="flex items-center space-x-2">
                      <span className="px-2 py-1 bg-gray-100 text-gray-800 rounded-full text-xs font-medium">
                        {language.code.toUpperCase()}
                      </span>
                    </div>
                  </div>
                  <div className="flex justify-between items-center">
                    <span className="flex items-center text-sm text-gray-500">
                      <X size={16} className="mr-1" />
                      Inactive
                    </span>
                    <div className="flex space-x-2">
                      <button
                        onClick={() => toggleLanguageStatus(language.id)}
                        className="text-green-600 hover:text-green-800 text-sm"
                      >
                        Activate
                      </button>
                      <button className="text-blue-600 hover:text-blue-800 p-1">
                        <Edit size={14} />
                      </button>
                      <button className="text-red-600 hover:text-red-800 p-1">
                        <Trash2 size={14} />
                      </button>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </div>
        </div>
      )}

      {/* <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
        <div className="flex">
          <div className="flex-shrink-0">
            <Check className="h-5 w-5 text-blue-400" />
          </div>
          <div className="ml-3">
            <h3 className="text-sm font-medium text-blue-800">Language Management</h3>
            <div className="mt-2 text-sm text-blue-700">
              <p>
                Only active languages will be displayed in frontend forms and available for users to select. 
                You can activate or deactivate languages as needed to control availability.
              </p>
            </div>
          </div>
        </div>
      </div> */}
    </div>
  );
}