"use client";

import { useState, useEffect } from "react";
import {
  Search,
  Filter,
  Star,
  Eye,
  Edit,
  CircleCheck,
  CircleX,
  CreditCard,
} from "lucide-react";
import "./membership.css";
interface Freelancer {
  id: string;
  email: string;
  profile: {
    firstName?: string;
    lastName?: string;
    title?: string;
    hourlyRate?: number;
    skills?: string[];
    profileImage?: string;
    country: {
      name: string;
      domain: string;
      currencySymbol: string;
    };
  };
  avgRating: number;
  totalProjects: number;
  completedProjects: number;
  isVerified: boolean;
  status: boolean;
  createdAt: string;
}

interface DetailedFreelancer {
  id: string;
  email: string;
  role: string;
  status: boolean;
  createdAt: string;
  profile?: {
    firstName?: string;
    lastName?: string;
    title?: string;
    hourlyRate?: number;
    phone?: string;
    city?: string;
    bio?: string;
    skills?: string[];
    country?: {
      name: string;
      domain: string;
      currency?: string;
      currencySymbol?: string;
    };
    experience?: Array<{
      title: string;
      company: string;
      location?: string;
      startDate: string;
      endDate?: string;
      isCurrent: boolean;
      description?: string;
    }>;
    education?: Array<{
      degree: string;
      institution: string;
      location?: string;
      startDate: string;
      endDate?: string;
      isCurrent: boolean;
      description?: string;
    }>;
    portfolio?: Array<{
      imageUrl: string;
      title: string;
      description?: string;
      projectUrl?: string;
    }>;
  };
  stats?: {
    totalProjects: number;
    totalBids: number;
    totalMessages: number;
    avgRating: number;
    totalReviews: number;
  };
  subscriptions?: Array<{
    package: {
      name: string;
      price: number;
      duration: number;
    };
    status: string;
    startDate: string;
    endDate: string;
  }>;
}

export default function FreelancersPage() {
  const [freelancers, setFreelancers] = useState<Freelancer[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState("");
  const [selectedFreelancer, setSelectedFreelancer] =
    useState<DetailedFreelancer | null>(null);
  const [showFreelancerModal, setShowFreelancerModal] = useState(false);
  const [membershipHistory, setMembershipHistory] = useState<any[]>([]);
  const [showMembershipModal, setShowMembershipModal] = useState(false);

  const handleViewMembership = async (freelancerId: string) => {
    try {
      const response = await fetch(
        `/api/freelancers/memberships/history?userId=${freelancerId}`
      );
      if (response.ok) {
        const data = await response.json();
        setMembershipHistory(data.data.transactions);
        setShowMembershipModal(true);
      } else {
        console.error("Failed to fetch membership history");
      }
    } catch (error) {
      console.error("Error fetching membership history:", error);
    }
  };

  const handleViewFreelancer = async (freelancerId: string) => {
    try {
      const response = await fetch(`/api/users/${freelancerId}`);
      if (response.ok) {
        const userData = await response.json();
        setSelectedFreelancer(userData);
        setShowFreelancerModal(true);
      } else {
        console.error("Failed to fetch freelancer details");
      }
    } catch (error) {
      console.error("Error fetching freelancer details:", error);
    }
  };

  const handleToggleFreelancerStatus = async (
    freelancerId: string,
    currentStatus: boolean
  ) => {
    try {
      const response = await fetch(`/api/users/${freelancerId}`, {
        method: "PUT",
        headers: {
          "Content-Type": "application/json",
        },
        body: JSON.stringify({
          status: !currentStatus,
        }),
      });

      if (response.ok) {
        // Update the freelancer in the local state
        setFreelancers((prevFreelancers) =>
          prevFreelancers.map((freelancer) =>
            freelancer.id === freelancerId
              ? { ...freelancer, status: !currentStatus }
              : freelancer
          )
        );
      } else {
        const errorData = await response.json();
        console.error("Failed to update freelancer status:", errorData.error);
        alert("Failed to update freelancer status: " + errorData.error);
      }
    } catch (error) {
      console.error("Error updating freelancer status:", error);
      alert("Error updating freelancer status");
    }
  };

  useEffect(() => {
    const fetchFreelancers = async () => {
      try {
        const response = await fetch("/api/freelancers");
        if (response.ok) {
          const data = await response.json();
          setFreelancers(data);
        } else {
          console.error("Failed to fetch freelancers");
          setFreelancers([]);
        }
      } catch (error) {
        console.error("Error fetching freelancers:", error);
        setFreelancers([]);
      } finally {
        setLoading(false);
      }
    };

    fetchFreelancers();
  }, []);

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-500"></div>
      </div>
    );
  }

  const filteredFreelancers = freelancers.filter(
    (freelancer) =>
      freelancer.email.toLowerCase().includes(searchTerm.toLowerCase()) ||
      `${freelancer.profile.firstName} ${freelancer.profile.lastName}`
        .toLowerCase()
        .includes(searchTerm.toLowerCase()) ||
      freelancer.profile.title?.toLowerCase().includes(searchTerm.toLowerCase())
  );
  const getStatusBadge = (status: string) => {
    switch (status) {
      case "active":
        return <span className="badge badge-active">Active</span>;
      case "expired":
        return <span className="badge badge-expired">Expired</span>;
      default:
        return <span className="badge badge-pending">Pending</span>;
    }
  };

  console.log("membershipHistory",membershipHistory)
  return (
    <div className="space-y-6 max-lg:space-y-4">
      <div>
        <h1 className="text-2xl font-semibold text-gray-900 max-lg:text-xl">Freelancers</h1>
        {/* <p className="text-gray-600 mt-2">Manage freelancer accounts and profiles</p> */}
      </div>

      {/* Search and Filters */}
      <div className="bg-white rounded-lg shadow p-4">
        <div className="flex flex-col sm:flex-row gap-4">
          <div className="flex-1">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
              <input
                type="text"
                placeholder="Search freelancers..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              />
            </div>
          </div>
          <div className="flex gap-2">
            <button className="flex items-center px-4 py-2 text-gray-700 bg-white border border-gray-300 rounded-lg hover:bg-gray-50">
              <Filter className="w-4 h-4 mr-2" />
              Filter
            </button>
          </div>
        </div>
      </div>

      {/* Freelancers Grid */}
      <div className="grid gap-6 md:grid-cols-2 lg:grid-cols-3 max-md:gap-4">
        {filteredFreelancers.map((freelancer) => (
          <div
            key={freelancer.id}
            className="bg-white rounded-lg shadow hover:shadow-md transition-shadow"
          >
            <div className="p-6 max-md:p-4">
              <div className="flex items-start justify-between mb-4">
                <div className="flex items-center space-x-3">
                  <div className="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center">
                    {freelancer.profile.profileImage ? (
                      <img
                        src={freelancer.profile.profileImage}
                        alt="Profile"
                        className="w-12 h-12 rounded-full object-cover"
                      />
                    ) : (
                      <span className="text-blue-600 font-semibold text-lg">
                        {freelancer.profile.firstName?.[0] ||
                          freelancer.email[0].toUpperCase()}
                      </span>
                    )}
                  </div>
                  <div className="w-[calc(100%-55px)]">
                    <h3 className="font-semibold text-gray-900">
                      {freelancer.profile.firstName &&
                      freelancer.profile.lastName
                        ? `${freelancer.profile.firstName} ${freelancer.profile.lastName}`
                        : freelancer.email}
                    </h3>
                    <p className="text-sm text-gray-600">
                      {freelancer.profile.title || "No title"}
                    </p>
                  </div>
                </div>
                {freelancer.isVerified && (
                  <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">
                    Verified
                  </span>
                )}
              </div>

              <div className="space-y-2 mb-4">
                <div className="flex items-center justify-between text-sm">
                  <span className="text-gray-600">Rating:</span>
                  <div className="flex items-center">
                    <Star className="w-4 h-4 text-yellow-400 fill-current" />
                    <span className="ml-1 text-gray-900">
                      {freelancer.avgRating || "N/A"}
                    </span>
                  </div>
                </div>
                <div className="flex items-center justify-between text-sm">
                  <span className="text-gray-600">Projects:</span>
                  <span className="text-gray-900">
                    {freelancer.completedProjects}/{freelancer.totalProjects}
                  </span>
                </div>
                {freelancer.profile.hourlyRate && (
                  <div className="flex items-center justify-between text-sm">
                    <span className="text-gray-600">Rate:</span>
                    <span className="text-gray-900">
                      {freelancer.profile.country?.currencySymbol || "$"}
                      {freelancer.profile.hourlyRate}/hr
                    </span>
                  </div>
                )}
                <div className="flex items-center justify-between text-sm">
                  <span className="text-gray-600">Country:</span>
                  <span className="text-gray-900">
                    {freelancer.profile.country?.name || "N/A"}
                  </span>
                </div>
              </div>

              {freelancer.profile.skills &&
                freelancer.profile.skills.length > 0 && (
                  <div className="mb-4">
                    <div className="flex flex-wrap gap-1">
                      {freelancer.profile.skills
                        .slice(0, 3)
                        .map((skill, index) => (
                          <span
                            key={index}
                            className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-blue-100 text-blue-800"
                          >
                            {skill}
                          </span>
                        ))}
                      {freelancer.profile.skills.length > 3 && (
                        <span className="text-xs text-gray-500">
                          +{freelancer.profile.skills.length - 3} more
                        </span>
                      )}
                    </div>
                  </div>
                )}

              <div className="flex items-center justify-between pt-4 border-t border-gray-200">
                <span className="text-xs text-gray-500">
                  Joined {new Date(freelancer.createdAt).toLocaleDateString()}
                </span>
                <div className="flex">
                  <button
                    onClick={() => handleViewFreelancer(freelancer.id)}
                    className="p-2 text-gray-400 hover:text-blue-600 transition-colors cursor-pointer"
                    title="View freelancer details"
                  >
                    <Eye className="w-6 h-6" />
                  </button>
                  <button
                    onClick={() => handleViewMembership(freelancer?.id)}
                    className="pl-0 p-2 text-purple-400 hover:text-gray-600 transition-colors cursor-pointer"
                    title="View membership history"
                  >
                    <CreditCard className="w-6 h-6" />
                  </button>
                  <button
                    onClick={() =>
                      handleToggleFreelancerStatus(
                        freelancer.id,
                        freelancer.status
                      )
                    }
                    className={`transition-colors cursor-pointer ${
                      freelancer.status
                        ? "text-green-400 hover:text-green-600"
                        : "text-red-400 hover:text-red-600"
                    }`}
                    title={
                      freelancer.status
                        ? "Deactivate freelancer"
                        : "Activate freelancer"
                    }
                  >
                    {freelancer.status ? (
                      <CircleCheck className="w-6 h-6" />
                    ) : (
                      <CircleX className="w-6 h-6" />
                    )}
                  </button>
                </div>
              </div>
            </div>
          </div>
        ))}
      </div>

      {filteredFreelancers.length === 0 && !loading && (
        <div className="text-center py-12">
          <div className="text-gray-400 text-lg mb-2">No freelancers found</div>
          <div className="text-gray-500 text-sm">
            {searchTerm
              ? "Try adjusting your search criteria"
              : "No freelancers have registered yet"}
          </div>
        </div>
      )}

      {/* Freelancer Details Modal */}
      {showFreelancerModal && selectedFreelancer && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 modal-wrapper">
          <div className="bg-white rounded-lg shadow-xl max-w-4xl w-full m-4 max-h-screen overflow-y-auto">
            <div className="flex justify-between items-center px-6 py-4 border-b border-gray-200 max-md:px-4">
              <h2 className="text-xl font-semibold">Freelancer Details</h2>
              <button
                  onClick={() => setShowFreelancerModal(false)}
                  className="text-gray-500 hover:text-gray-700"
                >
                  <svg
                    className="w-6 h-6"
                    fill="none"
                    stroke="currentColor"
                    viewBox="0 0 24 24"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      strokeWidth="2"
                      d="M6 18L18 6M6 6l12 12"
                    />
                  </svg>
                </button>
            </div>
            <div className="p-6 max-md:p-4">
              <div className="space-y-6">
                {/* Basic Info */}
                <div>
                  <h3 className="text-lg font-semibold text-gray-900 mb-3">
                    Basic Information
                  </h3>
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <p className="text-sm font-medium text-gray-500">Name</p>
                      <p className="text-sm text-gray-900">
                        {selectedFreelancer.profile?.firstName &&
                        selectedFreelancer.profile?.lastName
                          ? `${selectedFreelancer.profile.firstName} ${selectedFreelancer.profile.lastName}`
                          : "No name set"}
                      </p>
                    </div>
                    <div>
                      <p className="text-sm font-medium text-gray-500">Email</p>
                      <p className="text-sm text-gray-900">
                        {selectedFreelancer.email}
                      </p>
                    </div>
                    <div>
                      <p className="text-sm font-medium text-gray-500">Role</p>
                      <span className="px-2 py-1 rounded-full text-xs font-medium bg-green-100 text-green-800">
                        {selectedFreelancer.role}
                      </span>
                    </div>
                    <div>
                      <p className="text-sm font-medium text-gray-500">
                        Status
                      </p>
                      <span
                        className={`px-2 py-1 rounded-full text-xs font-medium ${
                          selectedFreelancer.status
                            ? "bg-green-100 text-green-800"
                            : "bg-red-100 text-red-800"
                        }`}
                      >
                        {selectedFreelancer.status ? "Active" : "Inactive"}
                      </span>
                    </div>
                    <div>
                      <p className="text-sm font-medium text-gray-500">
                        Country
                      </p>
                      <p className="text-sm text-gray-900">
                        {selectedFreelancer.profile?.country?.name ||
                          "Not specified"}
                      </p>
                    </div>
                    <div>
                      <p className="text-sm font-medium text-gray-500">
                        Joined
                      </p>
                      <p className="text-sm text-gray-900">
                        {new Date(
                          selectedFreelancer.createdAt
                        ).toLocaleDateString()}
                      </p>
                    </div>
                  </div>
                </div>

                {/* Freelancer-specific Info */}
                {selectedFreelancer.profile && (
                  <div>
                    <h3 className="text-lg font-semibold text-gray-900 mb-3">
                      Professional Information
                    </h3>
                    <div className="grid grid-cols-2 gap-4">
                      {selectedFreelancer.profile.title && (
                        <div>
                          <p className="text-sm font-medium text-gray-500">
                            Title
                          </p>
                          <p className="text-sm text-gray-900">
                            {selectedFreelancer.profile.title}
                          </p>
                        </div>
                      )}
                      {selectedFreelancer.profile.hourlyRate && (
                        <div>
                          <p className="text-sm font-medium text-gray-500">
                            Hourly Rate
                          </p>
                          <p className="text-sm text-gray-900">
                            {selectedFreelancer.profile.country
                              ?.currencySymbol || "$"}
                            {selectedFreelancer.profile.hourlyRate}
                          </p>
                        </div>
                      )}
                      {selectedFreelancer.profile.phone && (
                        <div>
                          <p className="text-sm font-medium text-gray-500">
                            Phone
                          </p>
                          <p className="text-sm text-gray-900">
                            {selectedFreelancer.profile.phone}
                          </p>
                        </div>
                      )}
                      {selectedFreelancer.profile.city && (
                        <div>
                          <p className="text-sm font-medium text-gray-500">
                            City
                          </p>
                          <p className="text-sm text-gray-900">
                            {selectedFreelancer.profile.city}
                          </p>
                        </div>
                      )}
                      {selectedFreelancer.profile.bio && (
                        <div className="col-span-2">
                          <p className="text-sm font-medium text-gray-500">
                            Bio
                          </p>
                          <p className="text-sm text-gray-900">
                            {selectedFreelancer.profile.bio}
                          </p>
                        </div>
                      )}
                      {selectedFreelancer.profile.skills &&
                        selectedFreelancer.profile.skills.length > 0 && (
                          <div className="col-span-2">
                            <p className="text-sm font-medium text-gray-500">
                              Skills
                            </p>
                            <div className="flex flex-wrap gap-2 mt-1">
                              {selectedFreelancer.profile.skills.map(
                                (skill, index) => (
                                  <span
                                    key={index}
                                    className="px-2 py-1 bg-blue-100 text-blue-800 text-xs rounded-full"
                                  >
                                    {skill}
                                  </span>
                                )
                              )}
                            </div>
                          </div>
                        )}
                    </div>
                  </div>
                )}

                {/* Statistics */}
                {selectedFreelancer.stats && (
                  <div>
                    <h3 className="text-lg font-semibold text-gray-900 mb-3">
                      Statistics
                    </h3>
                    <div className="grid grid-cols-2 gap-4">
                      <div>
                        <p className="text-sm font-medium text-gray-500">
                          Total Projects
                        </p>
                        <p className="text-sm text-gray-900">
                          {selectedFreelancer.stats.totalProjects}
                        </p>
                      </div>
                      <div>
                        <p className="text-sm font-medium text-gray-500">
                          Total Bids
                        </p>
                        <p className="text-sm text-gray-900">
                          {selectedFreelancer.stats.totalBids}
                        </p>
                      </div>
                      <div>
                        <p className="text-sm font-medium text-gray-500">
                          Average Rating
                        </p>
                        <p className="text-sm text-gray-900">
                          {selectedFreelancer.stats.avgRating > 0
                            ? `${selectedFreelancer.stats.avgRating.toFixed(
                                1
                              )}/5 (${
                                selectedFreelancer.stats.totalReviews
                              } reviews)`
                            : "No reviews yet"}
                        </p>
                      </div>
                      <div>
                        <p className="text-sm font-medium text-gray-500">
                          Total Messages
                        </p>
                        <p className="text-sm text-gray-900">
                          {selectedFreelancer.stats.totalMessages}
                        </p>
                      </div>
                    </div>
                  </div>
                )}

                {/* Experience */}
                {selectedFreelancer.profile?.experience &&
                  selectedFreelancer.profile.experience.length > 0 && (
                    <div>
                      <h3 className="text-lg font-semibold text-gray-900 mb-3">
                        Experience
                      </h3>
                      <div className="space-y-3">
                        {selectedFreelancer.profile.experience.map(
                          (exp, index) => (
                            <div
                              key={index}
                              className="p-3 bg-gray-50 rounded-lg"
                            >
                              <div className="flex justify-between items-start">
                                <div>
                                  <p className="font-medium text-gray-900">
                                    {exp.title}
                                  </p>
                                  <p className="text-sm text-gray-600">
                                    {exp.company}
                                  </p>
                                  {exp.location && (
                                    <p className="text-sm text-gray-500">
                                      {exp.location}
                                    </p>
                                  )}
                                </div>
                                <span className="text-xs text-gray-500">
                                  {new Date(exp.startDate).toLocaleDateString()}{" "}
                                  -{" "}
                                  {exp.isCurrent
                                    ? "Present"
                                    : exp.endDate
                                    ? new Date(exp.endDate).toLocaleDateString()
                                    : "N/A"}
                                </span>
                              </div>
                              {exp.description && (
                                <p className="text-sm text-gray-700 mt-2">
                                  {exp.description}
                                </p>
                              )}
                            </div>
                          )
                        )}
                      </div>
                    </div>
                  )}

                {/* Education */}
                {selectedFreelancer.profile?.education &&
                  selectedFreelancer.profile.education.length > 0 && (
                    <div>
                      <h3 className="text-lg font-semibold text-gray-900 mb-3">
                        Education
                      </h3>
                      <div className="space-y-3">
                        {selectedFreelancer.profile.education.map(
                          (edu, index) => (
                            <div
                              key={index}
                              className="p-3 bg-gray-50 rounded-lg"
                            >
                              <div className="flex justify-between items-start">
                                <div>
                                  <p className="font-medium text-gray-900">
                                    {edu.degree}
                                  </p>
                                  <p className="text-sm text-gray-600">
                                    {edu.institution}
                                  </p>
                                  {edu.location && (
                                    <p className="text-sm text-gray-500">
                                      {edu.location}
                                    </p>
                                  )}
                                </div>
                                <span className="text-xs text-gray-500">
                                  {new Date(edu.startDate).toLocaleDateString()}{" "}
                                  -{" "}
                                  {edu.isCurrent
                                    ? "Present"
                                    : edu.endDate
                                    ? new Date(edu.endDate).toLocaleDateString()
                                    : "N/A"}
                                </span>
                              </div>
                              {edu.description && (
                                <p className="text-sm text-gray-700 mt-2">
                                  {edu.description}
                                </p>
                              )}
                            </div>
                          )
                        )}
                      </div>
                    </div>
                  )}

                {/* Portfolio */}
                {selectedFreelancer.profile?.portfolio &&
                  selectedFreelancer.profile.portfolio.length > 0 && (
                    <div>
                      <h3 className="text-lg font-semibold text-gray-900 mb-3">
                        Portfolio
                      </h3>
                      <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        {selectedFreelancer.profile.portfolio.map(
                          (item, index) => (
                            <div
                              key={index}
                              className="border border-gray-200 rounded-lg overflow-hidden"
                            >
                              <img
                                src={item.imageUrl}
                                alt={item.title}
                                className="w-full h-32 object-cover"
                              />
                              <div className="p-3">
                                <p className="font-medium text-gray-900">
                                  {item.title}
                                </p>
                                {item.description && (
                                  <p className="text-sm text-gray-600 mt-1">
                                    {item.description}
                                  </p>
                                )}
                                {item.projectUrl && (
                                  <a
                                    href={item.projectUrl}
                                    target="_blank"
                                    rel="noopener noreferrer"
                                    className="text-sm text-blue-600 hover:text-blue-800 mt-2 inline-block"
                                  >
                                    View Project →
                                  </a>
                                )}
                              </div>
                            </div>
                          )
                        )}
                      </div>
                    </div>
                  )}

                {/* Subscriptions */}
                {selectedFreelancer.subscriptions &&
                  selectedFreelancer.subscriptions.length > 0 && (
                    <div>
                      <h3 className="text-lg font-semibold text-gray-900 mb-3">
                        Subscriptions
                      </h3>
                      <div className="space-y-2">
                        {selectedFreelancer.subscriptions.map(
                          (subscription, index) => (
                            <div
                              key={index}
                              className="p-3 bg-gray-50 rounded-lg"
                            >
                              <div className="flex justify-between items-center">
                                <div>
                                  <p className="font-medium text-gray-900">
                                    {subscription.package.name}
                                  </p>
                                  <p className="text-sm text-gray-500">
                                    {subscription.package.price}{" "}
                                    {selectedFreelancer.profile?.country
                                      ?.currency || "USD"}{" "}
                                    / {subscription.package.duration} days
                                  </p>
                                </div>
                                <span
                                  className={`px-2 py-1 rounded-full text-xs font-medium ${
                                    subscription.status === "ACTIVE"
                                      ? "bg-green-100 text-green-800"
                                      : "bg-gray-100 text-gray-800"
                                  }`}
                                >
                                  {subscription.status}
                                </span>
                              </div>
                              <p className="text-xs text-gray-500 mt-1">
                                {new Date(
                                  subscription.startDate
                                ).toLocaleDateString()}{" "}
                                -{" "}
                                {new Date(
                                  subscription.endDate
                                ).toLocaleDateString()}
                              </p>
                            </div>
                          )
                        )}
                      </div>
                    </div>
                  )}
              </div>
            </div>
            
            <div className="flex justify-end items-center px-6 py-4 border-t border-gray-200 max-md:px-4">
              <div className="flex justify-end">
                <button onClick={() => setShowFreelancerModal(false)} className="px-4 py-2 bg-gray-500 text-white rounded-md hover:bg-gray-600">
                  Close
                </button>
              </div>
            </div>
          </div>
        </div>
      )}
      {showMembershipModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4 modal-wrapper">
          <div className="bg-white rounded-lg shadow-lg w-full max-w-3xl overflow-hidden">
            <div className="flex justify-between items-center px-6 py-4 border-b border-gray-400">
              <h2 className="text-xl font-semibold">Membership History</h2>
              <button
                onClick={() => setShowMembershipModal(false)}
                className="text-gray-500 hover:text-gray-800 text-lg font-bold"
              >
                ✕
              </button>
            </div>

            <div className="p-6 overflow-x-auto max-md:p-4">
              {membershipHistory.length > 0 ? (
                <table className="min-w-full border border-gray-200">
                  <thead className="bg-gray-50">
                    <tr>
                      <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">
                        S.no
                      </th>
                      <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">
                        Category
                      </th>
                      <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">
                        Transaction ID
                      </th>
                      <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">
                        Start Date
                      </th>
                      <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">
                        End Date
                      </th>
                      <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">
                        Status
                      </th>
                      <th className="px-3 py-2 text-left text-xs font-medium text-gray-500 uppercase">
                        Payment
                      </th>
                    </tr>
                  </thead>
                  <tbody>
                    {membershipHistory.map((history, index) => (
                      <tr
                        key={history.id}
                        className="border-b hover:bg-gray-50"
                      >
                        <td className="px-3 py-2 text-sm text-gray-700">
                          {index + 1}
                        </td>
                        <td className="px-3 py-2 text-sm text-gray-700 flex items-center gap-2">
                          {history?.categoryIcon && (
                            <img
                              src={`/assets/image/category/${history?.categoryIcon}`}
                              alt={history?.category}
                              className="w-5 h-5"
                              onError={(e) => {
                                const target = e.target as HTMLImageElement;
                                target.src = `/assets/image/category/default.svg`;
                              }}
                            />
                          )}
                          {history.category}
                        </td>
                        <td className="px-3 py-2 text-sm text-gray-700">
                          {history?.transactionId}
                        </td>
                        <td className="px-3 py-2 text-sm text-gray-700">
                          {new Date(history?.startDate).toLocaleDateString()}
                        </td>
                        <td className="px-3 py-2 text-sm text-gray-700">
                          {new Date(history?.endDate).toLocaleDateString()}
                        </td>
                        <td className="px-3 py-2">
                          {getStatusBadge(history?.status)}
                        </td>
                        <td className="px-3 py-2 text-sm text-gray-700">
                          {history?.paymentStatus === "completed"
                            ? "Succeeded"
                            : "Failed"}
                        </td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              ) : (
                <p className="text-gray-500">
                  No memberships found for this freelancer.
                </p>
              )}
            </div>
          </div>
        </div>
      )}
    </div>
  );
}
