"use client";

import { useState, useEffect } from "react";
import { Edit2 } from "lucide-react";
import DomainModal from "@/components/DomainModal";
import { showSuccess, showError } from "@/helpers/toastMessage";
import axios from "axios";
interface Domain {
  id: string;
  name: string;
  code: string;
  domain: string;
  languages: string[];
  defaultLanguage: string;
  currency: string;
  currencySymbol: string;
  packagePrice: number;
  paymentGateways: string[];
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
  businessEmail?: string;
  facebookUrl?: string;
  googleUrl?: string;
  instagramUrl?: string;
  linkedinUrl?: string;
  location?: string;
  subLocation?: string;
  logoFooter?: string;
  logoHeader?: string;
  shortDescription?: string;
  supportEmail?: string;
  twitterUrl?: string;
}

// Safely parse a JSON string to array
const safeParseArray = (value: any): string[] => {
  if (!value) return [];
  if (Array.isArray(value)) return value;
  try {
    return JSON.parse(value);
  } catch {
    return [];
  }
};

export default function DomainsPage() {
  const [domains, setDomains] = useState<Domain[]>([]);
  const [loading, setLoading] = useState(true);
  const [showEditModal, setShowEditModal] = useState(false);
  const [currentDomainData, setCurrentDomainData] = useState<Domain | null>(null);

useEffect(() => {
  const fetchDomains = async () => {
    try {
      const response = await axios.get("/api/countries");

      const data = response.data;

      const normalizedData: Domain[] = data.map((d: any) => ({
        ...d,
        languages: safeParseArray(d.languages),
        paymentGateways: safeParseArray(d.paymentGateways),
      }));

      setDomains(normalizedData);
    } catch (err) {
      console.error(err);
      setDomains([]);
    } finally {
      setLoading(false);
    }
  };

  fetchDomains();
}, []);
  // Open modal with selected domain
  const openDomainModal = (code: string) => {
    const selected = domains.find((d) => d.code === code) || null;
    setCurrentDomainData(selected);
    setShowEditModal(true);
  };

  // Update domain in the list instantly
  const handleDomainUpdate = (updatedDomain: Domain) => {
    setDomains((prev) =>
      prev.map((d) => (d.id === updatedDomain.id ? updatedDomain : d))
    );
    setShowEditModal(false);
  };

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-500"></div>
      </div>
    );
  }

  return (
    <div className="space-y-6 max-lg:space-y-4">
      <div className="flex justify-between items-center">
        <h1 className="text-2xl font-semibold text-gray-900 max-lg:text-xl">
          Domains
        </h1>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 gap-6 max-lg:!grid-cols-1">
        {domains.map((domain) => (
          <div
            key={domain.id}
            className="bg-white rounded-lg shadow border border-gray-100 hover:shadow-lg transition-shadow duration-200"
          >
            <div className="p-6 max-md:p-4">
              <div className="flex justify-between items-start mb-4 space-x-1 flex-wrap max-md:gap-y-2">
                <div>
                  <h3 className="text-lg font-medium text-gray-900">{domain.name}</h3>
                  <p className="text-blue-600 font-medium">{domain.domain}</p>
                </div>

                <div>
                  <span className="text-gray-600 text-base max-lg:text-sm">Payment Gateways:</span>
                  <div className="flex flex-wrap gap-1 mt-1">
                    {safeParseArray(domain.paymentGateways).map((gateway) => (
                      <span
                        key={gateway}
                        className="px-2 py-1 bg-purple-100 text-purple-800 rounded text-xs"
                      >
                        {gateway}
                      </span>
                    ))}
                  </div>
                </div>

                <div>
                  <span className="text-gray-600 text-base max-lg:text-sm">Currency:</span>
                  <span className="ml-2 font-medium text-base max-lg:text-sm">{domain.currency}</span>
                </div>

                <div>
                  <span className="text-gray-600 text-base max-lg:text-sm">Price:</span>
                  <span className="ml-2 font-medium text-base max-lg:text-sm">
                    {domain.currencySymbol}{domain.packagePrice}
                  </span>
                </div>
              </div>

              <div className="space-y-3">
                <div>
                  <span className="text-gray-600 text-sm">Languages:</span>
                  <div className="flex flex-wrap gap-1 mt-1 items-center">
                    {safeParseArray(domain.languages).map((lang) => (
                      <span
                        key={lang}
                        className={`px-2 py-1 rounded text-xs ${
                          lang === domain.defaultLanguage
                            ? "bg-blue-100 text-blue-800"
                            : "bg-gray-100 text-gray-700"
                        }`}
                      >
                        {lang.toUpperCase()}
                        {lang === domain.defaultLanguage && " (default)"}
                      </span>
                    ))}

                    <button
                      type="button"
                      className="text-xs font-medium flex items-center gap-1 text-blue-500 hover:text-blue-700 px-2 py-1 bg-blue-100 rounded-sm"
                      onClick={() => openDomainModal(domain.code)}
                    >
                      <Edit2 size={14} />
                      Edit
                    </button>
                  </div>
                </div>
              </div>
            </div>
          </div>
        ))}
      </div>

      {showEditModal && currentDomainData && (
        <DomainModal
          showEditModal={showEditModal}
          setShowEditModal={setShowEditModal}
          currentDomainData={currentDomainData}
          onDomainUpdated={handleDomainUpdate}
        />
      )}
    </div>
  );
}
