'use client';

import { useState, useEffect } from 'react';
import { 
  Users, 
  Briefcase, 
  UserCheck, 
  DollarSign,
  TrendingUp,
  Calendar
} from 'lucide-react';
import { BarChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer, LineChart, Line } from 'recharts';

interface DashboardStats {
  totalUsers: number;
  totalProjects: number;
  totalFreelancers: number;
  totalRevenue: number;
  activeSubscriptions: number;
  monthlyGrowth: {
    users: number;
    projects: number;
    freelancers: number;
  };
}

const StatCard = ({ title, value, icon: Icon, change }: {
  title: string;
  value: string | number;
  icon: React.ComponentType<{ size?: number; className?: string }>;
  change?: string;
}) => (
  <div className="bg-white rounded-lg shadow p-6 max-xl:p-4">
    <div className="flex items-center justify-between">
      <div>
        <p className="text-sm font-medium text-gray-600">{title}</p>
        <p className="text-2xl font-bold text-gray-900">{value}</p>
        {change && (
          <p className="text-sm text-green-600 flex items-center mt-1">
            <TrendingUp size={14} className="mr-1" />
            {change}
          </p>
        )}
      </div>
      <div className="w-12 h-12 bg-blue-50 rounded-lg flex items-center justify-center">
        <Icon size={24} className="text-blue-500" />
      </div>
    </div>
  </div>
);

const FilterBar = ({ dateRange, setDateRange, domain, setDomain }: {
  dateRange: { from: string; to: string };
  setDateRange: (range: { from: string; to: string }) => void;
  domain: string;
  setDomain: (domain: string) => void;
}) => (
  <div className="bg-white rounded-lg shadow p-4 mb-6">
    <div className="flex flex-wrap gap-4 items-center">
      <div className="flex items-center space-x-2">
        <Calendar size={16} className="text-gray-400" />
        <label className="text-sm font-medium text-gray-700">From:</label>
        <input
          type="date"
          value={dateRange.from}
          onChange={(e) => setDateRange({ ...dateRange, from: e.target.value })}
          className="border border-gray-300 rounded px-3 py-1 text-sm"
        />
      </div>
      <div className="flex items-center space-x-2">
        <label className="text-sm font-medium text-gray-700">To:</label>
        <input
          type="date"
          value={dateRange.to}
          onChange={(e) => setDateRange({ ...dateRange, to: e.target.value })}
          className="border border-gray-300 rounded px-3 py-1 text-sm"
        />
      </div>
      <div className="flex items-center space-x-2">
        <label className="text-sm font-medium text-gray-700">Domain:</label>
        <select
          value={domain}
          onChange={(e) => setDomain(e.target.value)}
          className="border border-gray-300 rounded px-3 py-1 text-sm"
        >
          <option value="">All Domains</option>
          <option value="smile24.ch">smile24.ch</option>
          <option value="smile24.fr">smile24.fr</option>
          <option value="smile24.de">smile24.de</option>
          <option value="smile24.it">smile24.it</option>
        </select>
      </div>
    </div>
  </div>
);

interface ChartData {
  monthlyData: Array<{
    name: string;
    users: number;
    projects: number;
    revenue: number;
  }>;
  revenueData: Array<{
    name: string;
    revenue: number;
  }>;
}

interface Activity {
  action: string;
  user: string;
  time: string;
}

export default function Dashboard() {
  const [dateRange, setDateRange] = useState({
    from: new Date(new Date().getFullYear(), new Date().getMonth(), 1).toISOString().split('T')[0],
    to: new Date().toISOString().split('T')[0]
  });
  const [domain, setDomain] = useState('');
  const [stats, setStats] = useState<DashboardStats>({
    totalUsers: 0,
    totalProjects: 0,
    totalFreelancers: 0,
    totalRevenue: 0,
    activeSubscriptions: 0,
    monthlyGrowth: {
      users: 0,
      projects: 0,
      freelancers: 0
    }
  });
  const [chartData, setChartData] = useState<ChartData>({
    monthlyData: [],
    revenueData: []
  });
  const [activities, setActivities] = useState<Activity[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const fetchDashboardData = async () => {
      try {
        setLoading(true);
        
        // Fetch stats
        const statsResponse = await fetch('/api/dashboard/stats');
        if (statsResponse.ok) {
          const statsData = await statsResponse.json();
          setStats(statsData);
        }

        // Fetch chart data
        const chartParams = new URLSearchParams();
        if (dateRange.from) chartParams.set('from', dateRange.from);
        if (dateRange.to) chartParams.set('to', dateRange.to);
        if (domain) chartParams.set('domain', domain);

        const chartResponse = await fetch(`/api/dashboard/charts?${chartParams}`);
        if (chartResponse.ok) {
          const chartDataResponse = await chartResponse.json();
          setChartData(chartDataResponse);
        }

        // Fetch recent activities
        const activitiesResponse = await fetch('/api/dashboard/activities');
        if (activitiesResponse.ok) {
          const activitiesData = await activitiesResponse.json();
          setActivities(activitiesData);
        }

      } catch (error) {
        console.error('Error fetching dashboard data:', error);
      } finally {
        setLoading(false);
      }
    };

    fetchDashboardData();
  }, [dateRange, domain]);

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-500"></div>
      </div>
    );
  }

  return (
    <div className="space-y-6 max-lg:space-y-4">
      <div>
        <h1 className="text-2xl font-semibold text-gray-900 max-lg:text-xl">Dashboard</h1>
        <p className="text-gray-600 max-md:text-sm">Overview of your platform performance</p>
      </div>

      <FilterBar 
        dateRange={dateRange}
        setDateRange={setDateRange}
        domain={domain}
        setDomain={setDomain}
      />

      {/* Stats Grid */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 max-md:!grid-cols-2 max-sm:!grid-cols-1 max-xl:!grid-cols-3 max-lg:!grid-cols-2">
        <StatCard
          title="Total Users"
          value={stats.totalUsers.toLocaleString()}
          icon={Users}
          change={`+${stats.monthlyGrowth.users}% from last month`}
        />
        <StatCard
          title="Total Projects"
          value={stats.totalProjects.toLocaleString()}
          icon={Briefcase}
          change={`+${stats.monthlyGrowth.projects}% from last month`}
        />
        <StatCard
          title="Freelancers"
          value={stats.totalFreelancers.toLocaleString()}
          icon={UserCheck}
          change={`+${stats.monthlyGrowth.freelancers}% from last month`}
        />
        <StatCard
          title="Total Revenue"
          value={`$${stats.totalRevenue.toLocaleString()}`}
          icon={DollarSign}
          change={`${stats.activeSubscriptions} active subscriptions`}
        />
      </div>

      {/* Charts */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <div className="bg-white rounded-lg shadow p-6 max-md:p-4">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Monthly Overview</h3>
          <ResponsiveContainer width="100%" height={300}>
            <BarChart data={chartData.monthlyData}>
              <CartesianGrid strokeDasharray="3 3" />
              <XAxis dataKey="name" />
              <YAxis />
              <Tooltip />
              <Bar dataKey="users" fill="#3B82F6" name="Users" />
              <Bar dataKey="projects" fill="#10B981" name="Projects" />
            </BarChart>
          </ResponsiveContainer>
        </div>

        <div className="bg-white rounded-lg shadow p-6 max-md:p-4">
          <h3 className="text-lg font-semibold text-gray-900 mb-4">Revenue Trend</h3>
          <ResponsiveContainer width="100%" height={300}>
            <LineChart data={chartData.revenueData}>
              <CartesianGrid strokeDasharray="3 3" />
              <XAxis dataKey="name" />
              <YAxis />
              <Tooltip />
              <Line type="monotone" dataKey="revenue" stroke="#8B5CF6" strokeWidth={2} />
            </LineChart>
          </ResponsiveContainer>
        </div>
      </div>

      {/* Recent Activities */}
      <div className="bg-white rounded-lg shadow">
        <div className="p-6 py-4 border-b border-gray-200 max-md:p-4">
          <h3 className="text-lg font-semibold text-gray-900">Recent Activities</h3>
        </div>
        <div className="p-6 max-md:p-4">
          <div className="space-y-4">
            {activities.length > 0 ? (
              activities.map((activity, index) => (
                <div key={index} className="flex items-center justify-between pt-0 py-3 border-b border-gray-100 last:border-b-0">
                  <div>
                    <p className="text-sm font-medium text-gray-900">{activity.action}</p>
                    <p className="text-sm text-gray-600">{activity.user}</p>
                  </div>
                  <span className="text-xs text-gray-500">{activity.time}</span>
                </div>
              ))
            ) : (
              <p className="text-gray-500 text-center py-4">No recent activities</p>
            )}
          </div>
        </div>
      </div>
    </div>
  );
}