'use client';

import { useState, useEffect } from 'react';
import { Globe, Check, X } from 'lucide-react';

interface Country {
  id: string;
  name: string;
  code: string;
  domain: string;
  languages: string[];
  defaultLanguage: string;
  currency: string;
  currencySymbol: string;
  packagePrice: number;
  paymentGateways: string[];
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
}

export default function CountriesPage() {
  const [countries, setCountries] = useState<Country[]>([]);
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    const fetchCountries = async () => {
      try {
        const response = await fetch('/api/countries');
        if (response.ok) {
          const data = await response.json();
          setCountries(data);
        } else {
          console.error('Failed to fetch countries');
          setCountries([]);
        }
      } catch (error) {
        console.error('Error fetching countries:', error);
        setCountries([]);
      } finally {
        setLoading(false);
      }
    };

    fetchCountries();
  }, []);

  if (loading) {
    return <div className="flex items-center justify-center h-64">
      <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-500"></div>
    </div>;
  }

  const activeCountries = countries.filter(country => country.isActive);
  const inactiveCountries = countries.filter(country => !country.isActive);

  return (
    <div className="space-y-6 max-lg:space-y-4">
      <div className="flex justify-between items-center">
        <div>
          <h1 className="text-2xl font-semibold text-gray-900 max-lg:text-xl">Countries</h1>
        </div>
      </div>

      {/* Active Countries */}
      <div className="bg-white rounded-lg shadow">
        <div className="px-6 py-4 border-b border-gray-200 max-md:px-4">
          <h2 className="text-lg font-semibold text-gray-900">
            Active Countries ({activeCountries.length})
          </h2>
          <p className="text-sm text-gray-600">These countries are available for users</p>
        </div>
        <div className="p-6 max-md:p-4">
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
            {activeCountries.map((country) => (
              <div key={country.id} className="border border-green-200 bg-green-50 rounded-lg p-4">
                <div className="flex justify-between items-start mb-3">
                  <div>
                    <h3 className="font-semibold text-gray-900 flex items-center">
                      <Globe size={16} className="mr-2" />
                      {country.name}
                    </h3>
                    <p className="text-sm text-gray-600">{country.domain}</p>
                  </div>
                  <div className="flex items-center space-x-2">
                    <span className="px-2 py-1 bg-green-100 text-green-800 rounded-full text-xs font-medium">
                      {country.code.toUpperCase()}
                    </span>
                  </div>
                </div>
                <div className="space-y-2 text-sm">
                  <div className="flex justify-between">
                    <span className="text-gray-600">Currency:</span>
                    <span className="font-medium">{country.currencySymbol} {country.currency}</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-gray-600">Package Price:</span>
                    <span className="font-medium">{country.currencySymbol}{country.packagePrice}</span>
                  </div>
                  <div className="flex justify-between">
                    <span className="text-gray-600">Default Language:</span>
                    <span className="font-medium">{country.defaultLanguage}</span>
                  </div>
                  <div>
                    <span className="text-gray-600">Languages:</span>
                    <div className="flex flex-wrap gap-1 mt-1">
                      {country.languages.map((lang) => (
                        <span key={lang} className="px-2 py-1 bg-blue-100 text-blue-800 rounded text-xs">
                          {lang}
                        </span>
                      ))}
                    </div>
                  </div>
                  <div>
                    <span className="text-gray-600">Payment Gateways:</span>
                    <div className="flex flex-wrap gap-1 mt-1">
                      {country.paymentGateways.map((gateway) => (
                        <span key={gateway} className="px-2 py-1 bg-purple-100 text-purple-800 rounded text-xs">
                          {gateway}
                        </span>
                      ))}
                    </div>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* Inactive Countries */}
      {inactiveCountries.length > 0 && (
        <div className="bg-white rounded-lg shadow">
          <div className="px-6 py-4 border-b border-gray-200 max-md:px-4">
            <h2 className="text-lg font-semibold text-gray-900">
              Inactive Countries ({inactiveCountries.length})
            </h2>
            <p className="text-sm text-gray-600">These countries are not available for users</p>
          </div>
          <div className="p-6 max-md:p-4">
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
              {inactiveCountries.map((country) => (
                <div key={country.id} className="border border-gray-200 bg-gray-50 rounded-lg p-4">
                  <div className="flex justify-between items-start mb-3">
                    <div>
                      <h3 className="font-semibold text-gray-900 flex items-center">
                        <Globe size={16} className="mr-2" />
                        {country.name}
                      </h3>
                      <p className="text-sm text-gray-600">{country.domain}</p>
                    </div>
                    <div className="flex items-center space-x-2">
                      <span className="px-2 py-1 bg-gray-100 text-gray-800 rounded-full text-xs font-medium">
                        {country.code.toUpperCase()}
                      </span>
                    </div>
                  </div>
                  <div className="space-y-2 text-sm">
                    <div className="flex justify-between">
                      <span className="text-gray-600">Currency:</span>
                      <span className="font-medium">{country.currencySymbol} {country.currency}</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-gray-600">Package Price:</span>
                      <span className="font-medium">{country.currencySymbol}{country.packagePrice}</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-gray-600">Default Language:</span>
                      <span className="font-medium">{country.defaultLanguage}</span>
                    </div>
                    <div className="flex justify-between items-center">
                      <span className="flex items-center text-sm text-gray-500">
                        <X size={16} className="mr-1" />
                        Inactive
                      </span>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          </div>
        </div>
      )}
    </div>
  );
}