'use client';

import { useState, useEffect } from 'react';
import { Search, Eye, CircleX, CircleCheck } from 'lucide-react';

interface User {
  id: string;
  email: string;
  role: 'CLIENT' | 'FREELANCER' | 'ADMIN';
  profile?: {
    firstName?: string;
    lastName?: string;
    country?: {
      name: string;
      domain: string;
    };
  };
  subscriptions: Array<{
    status: 'ACTIVE' | 'EXPIRED' | 'CANCELLED';
    package: {
      name: string;
    };
  }>;
  status: boolean;
  createdAt: string;
  updatedAt: string;
}

interface DetailedUser {
  id: string;
  email: string;
  role: 'CLIENT' | 'FREELANCER' | 'ADMIN';
  profile?: {
    firstName?: string;
    lastName?: string;
    phone?: string;
    city?: string;
    bio?: string;
    country?: {
      name: string;
      domain: string;
      currency?: string;
    };
  };
  subscriptions: Array<{
    status: 'ACTIVE' | 'EXPIRED' | 'CANCELLED';
    package: {
      name: string;
      price: number;
      duration: number;
    };
    startDate: string;
    endDate: string;
  }>;
  stats?: {
    totalProjects: number;
    totalBids: number;
    totalMessages: number;
    avgRating: number;
    totalReviews: number;
  };
  status: boolean;
  createdAt: string;
  updatedAt: string;
}

const FilterBar = ({ 
  searchTerm, 
  setSearchTerm, 
  domainFilter,
  setDomainFilter
}: {
  searchTerm: string;
  setSearchTerm: (term: string) => void;
  domainFilter: string;
  setDomainFilter: (domain: string) => void;
}) => (
  <div className="bg-white rounded-lg shadow p-4 mb-6">
    <div className="flex flex-wrap gap-4 items-center">
      <div className="flex-1 min-w-64">
        <div className="relative">
          <Search size={16} className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400" />
          <input
            type="text"
            placeholder="Search users by name or email..."
            value={searchTerm}
            onChange={(e) => setSearchTerm(e.target.value)}
            className="w-full pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
          />
        </div>
      </div>
      
      <select
        value={domainFilter}
        onChange={(e) => setDomainFilter(e.target.value)}
        className="border border-gray-300 rounded-lg px-3 py-2 text-sm min-h-[42px]"
      >
        <option value="">All Domains</option>
        <option value="smile24.ch">smile24.ch</option>
        <option value="smile24.fr">smile24.fr</option>
        <option value="smile24.de">smile24.de</option>
        <option value="smile24.it">smile24.it</option>
      </select>
    </div>
  </div>
);

const UserRow = ({ 
  user, 
  onViewUser, 
  onToggleStatus 
}: { 
  user: User; 
  onViewUser: (userId: string) => void;
  onToggleStatus: (userId: string, currentStatus: boolean) => void;
}) => {
  const getStatusBadge = (role: string) => {
    const colors = {
      CLIENT: 'bg-blue-100 text-blue-800',
      FREELANCER: 'bg-green-100 text-green-800',
      ADMIN: 'bg-purple-100 text-purple-800',
    };
    return (
      <span className={`px-2 py-1 rounded-full text-xs font-medium ${colors[role as keyof typeof colors]}`}>
        {role}
      </span>
    );
  };

  const getSubscriptionStatus = () => {
    const activeSubscription = user.subscriptions.find(sub => sub.status === 'ACTIVE');
    if (activeSubscription) {
      return (
        <span className="px-2 py-1 rounded-full text-xs font-medium bg-green-100 text-green-800">
          {activeSubscription.package.name}
        </span>
      );
    }
    return (
      <span className="px-2 py-1 rounded-full text-xs font-medium bg-gray-100 text-gray-800">
        No active subscription
      </span>
    );
  };

  return (
    <tr className="hover:bg-gray-50">
      <td className="px-6 py-4 whitespace-nowrap">
        <div>
          <div className="text-sm font-medium text-gray-900">
            {user.profile?.firstName && user.profile?.lastName 
              ? `${user.profile.firstName} ${user.profile.lastName}`
              : 'No name set'
            }
          </div>
          <div className="text-sm text-gray-500">{user.email}</div>
        </div>
      </td>
      <td className="px-6 py-4 whitespace-nowrap">
        {getStatusBadge(user.role)}
      </td>
      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
        {user.profile?.country?.name || 'Not specified'}
      </td>
      <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
        {new Date(user.createdAt).toLocaleDateString()}
      </td>
      <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
        <div className="flex items-center justify-end space-x-2">
          <button 
            onClick={() => onViewUser(user.id)}
            className="text-blue-600 hover:text-blue-900 p-1 cursor-pointer"
            title="View user details"
          >
            <Eye size={24} />
          </button>
          <button 
            onClick={() => onToggleStatus(user.id, user.status)}
            className={`p-1 ${user.status ? 'text-green-600 hover:text-green-900 cursor-pointer' : 'text-red-600 hover:text-red-900 cursor-pointer'}`}
            title={user.status ? 'Deactivate user' : 'Activate user'}
          >
            {user.status ? <CircleCheck size={24} /> : <CircleX size={24} />}
          </button>
        </div>
      </td>
    </tr>
  );
};

export default function UsersPage() {
  const [users, setUsers] = useState<User[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [domainFilter, setDomainFilter] = useState('');
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const [selectedUser, setSelectedUser] = useState<DetailedUser | null>(null);
  const [showUserModal, setShowUserModal] = useState(false);
  const usersPerPage = 10;

  useEffect(() => {
    const fetchUsers = async () => {
      try {
        const params = new URLSearchParams({
          page: currentPage.toString(),
          limit: usersPerPage.toString()
        });
        params.set('role', 'CLIENT');
        if (domainFilter) params.set('domain', domainFilter);

        const response = await fetch(`/api/users?${params}`);
        if (response.ok) {
          const data = await response.json();
          setUsers(data.users);
          setTotalPages(data.totalPages);
        } else {
          console.error('Failed to fetch users');
          setUsers([]);
        }
      } catch (error) {
        console.error('Error fetching users:', error);
        setUsers([]);
      } finally {
        setLoading(false);
      }
    };

    fetchUsers();
  }, [domainFilter, currentPage]);

  const handleViewUser = async (userId: string) => {
    try {
      const response = await fetch(`/api/users/${userId}`);
      if (response.ok) {
        const userData = await response.json();
        setSelectedUser(userData);
        setShowUserModal(true);
      } else {
        console.error('Failed to fetch user details');
      }
    } catch (error) {
      console.error('Error fetching user details:', error);
    }
  };

  const handleToggleStatus = async (userId: string, currentStatus: boolean) => {
    try {
      const response = await fetch(`/api/users/${userId}`, {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({
          status: !currentStatus
        }),
      });

      if (response.ok) {
        // Update the user in the local state
        setUsers(prevUsers => 
          prevUsers.map(user => 
            user.id === userId 
              ? { ...user, status: !currentStatus }
              : user
          )
        );
      } else {
        const errorData = await response.json();
        console.error('Failed to update user status:', errorData.error);
        alert('Failed to update user status: ' + errorData.error);
      }
    } catch (error) {
      console.error('Error updating user status:', error);
      alert('Error updating user status');
    }
  };

  const filteredUsers = users.filter(user => {
    const matchesSearch = !searchTerm || 
      user.email.toLowerCase().includes(searchTerm.toLowerCase()) ||
      `${user.profile?.firstName} ${user.profile?.lastName}`.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesDomain = !domainFilter || user.profile?.country?.domain === domainFilter;
    
    return matchesSearch && matchesDomain;
  });

  const paginatedUsers = filteredUsers.slice(
    (currentPage - 1) * usersPerPage,
    currentPage * usersPerPage
  );

  if (loading) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-32 w-32 border-b-2 border-blue-500"></div>
      </div>
    );
  }

  return (
    <div className="space-y-6 max-lg:space-y-4">
      <div className="flex justify-between items-center">
        <div>
          <h1 className="text-2xl font-semibold text-gray-900 max-lg:text-xl">Clients</h1>
          {/* <p className="text-gray-600 mt-2">Manage platform users and their accounts</p> */}
        </div>
        {/* <button className="bg-blue-500 hover:bg-blue-600 text-white px-4 py-2 rounded-lg flex items-center space-x-2">
          <Plus size={16} />
          <span>Add User</span>
        </button> */}
      </div>

      <FilterBar
        searchTerm={searchTerm}
        setSearchTerm={setSearchTerm}
        domainFilter={domainFilter}
        setDomainFilter={setDomainFilter}
      />

      <div className="bg-white rounded-lg shadow overflow-hidden">
        <div className="overflow-x-auto">
          <table className="min-w-full divide-y divide-gray-200">
            <thead className="bg-gray-50">
              <tr>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  User
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Role
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Country
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Joined
                </th>
                <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Actions
                </th>
              </tr>
            </thead>
            <tbody className="bg-white divide-y divide-gray-200">
              {paginatedUsers.map((user) => (
                <UserRow 
                  key={user.id} 
                  user={user} 
                  onViewUser={handleViewUser}
                  onToggleStatus={handleToggleStatus}
                />
              ))}
            </tbody>
          </table>
        </div>
      </div>

      {/* Pagination */}
      <div className="flex items-center justify-between">
        <div className="text-sm text-gray-700">
          Showing {(currentPage - 1) * usersPerPage + 1} to {Math.min(currentPage * usersPerPage, filteredUsers.length)} of {filteredUsers.length} users
        </div>
        <div className="flex space-x-2">
          <button
            onClick={() => setCurrentPage(Math.max(1, currentPage - 1))}
            disabled={currentPage === 1}
            className="px-3 py-2 border border-gray-300 rounded-lg text-sm font-medium text-gray-700 hover:bg-gray-50 disabled:opacity-50"
          >
            Previous
          </button>
          <button
            onClick={() => setCurrentPage(Math.min(totalPages, currentPage + 1))}
            disabled={currentPage === totalPages}
            className="px-3 py-2 border border-gray-300 rounded-lg text-sm font-medium text-gray-700 hover:bg-gray-50 disabled:opacity-50"
          >
            Next
          </button>
        </div>
      </div>

      {/* User Details Modal */}
      {showUserModal && selectedUser && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 modal-wrapper">
          <div className="bg-white rounded-lg shadow-xl max-w-2xl w-full m-4 max-h-screen overflow-y-auto">
            <div className="flex justify-between items-center px-6 py-4 border-b border-gray-200 max-md:px-4">
              <h2 className="text-xl font-semibold">User Details</h2>
              <button
                  onClick={() => setShowUserModal(false)}
                  className="text-gray-500 hover:text-gray-700"
                >
                  <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M6 18L18 6M6 6l12 12" />
                  </svg>
                </button>
            </div>

            <div className="p-6 max-md:p-4">
             <div className="space-y-6">
                {/* Basic Info */}
                <div>
                  <h3 className="text-lg font-semibold text-gray-900 mb-3">Basic Information</h3>
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <p className="text-sm font-medium text-gray-500">Name</p>
                      <p className="text-sm text-gray-900">
                        {selectedUser.profile?.firstName && selectedUser.profile?.lastName 
                          ? `${selectedUser.profile.firstName} ${selectedUser.profile.lastName}`
                          : 'No name set'
                        }
                      </p>
                    </div>
                    <div>
                      <p className="text-sm font-medium text-gray-500">Email</p>
                      <p className="text-sm text-gray-900">{selectedUser.email}</p>
                    </div>
                    <div>
                      <p className="text-sm font-medium text-gray-500">Role</p>
                      <span className="px-2 py-1 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
                        {selectedUser.role}
                      </span>
                    </div>
                    <div>
                      <p className="text-sm font-medium text-gray-500">Status</p>
                      <span className={`px-2 py-1 rounded-full text-xs font-medium ${
                        selectedUser.status ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'
                      }`}>
                        {selectedUser.status ? 'Active' : 'Inactive'}
                      </span>
                    </div>
                    <div>
                      <p className="text-sm font-medium text-gray-500">Country</p>
                      <p className="text-sm text-gray-900">{selectedUser.profile?.country?.name || 'Not specified'}</p>
                    </div>
                    <div>
                      <p className="text-sm font-medium text-gray-500">Joined</p>
                      <p className="text-sm text-gray-900">
                        {new Date(selectedUser.createdAt).toLocaleDateString()}
                      </p>
                    </div>
                  </div>
                </div>

                {/* Profile Info */}
                {selectedUser.profile && (
                  <div>
                    <h3 className="text-lg font-semibold text-gray-900 mb-3">Profile Information</h3>
                    <div className="grid grid-cols-2 gap-4">
                      {selectedUser.profile.phone && (
                        <div>
                          <p className="text-sm font-medium text-gray-500">Phone</p>
                          <p className="text-sm text-gray-900">{selectedUser.profile.phone}</p>
                        </div>
                      )}
                      {selectedUser.profile.city && (
                        <div>
                          <p className="text-sm font-medium text-gray-500">City</p>
                          <p className="text-sm text-gray-900">{selectedUser.profile.city}</p>
                        </div>
                      )}
                      {selectedUser.profile.bio && (
                        <div className="col-span-2">
                          <p className="text-sm font-medium text-gray-500">Bio</p>
                          <p className="text-sm text-gray-900">{selectedUser.profile.bio}</p>
                        </div>
                      )}
                    </div>
                  </div>
                )}

                {/* Statistics */}
                {selectedUser.stats && (
                  <div>
                    <h3 className="text-lg font-semibold text-gray-900 mb-3">Statistics</h3>
                    <div className="grid grid-cols-2 gap-4">
                      <div>
                        <p className="text-sm font-medium text-gray-500">Total Projects</p>
                        <p className="text-sm text-gray-900">{selectedUser.stats.totalProjects}</p>
                      </div>
                      <div>
                        <p className="text-sm font-medium text-gray-500">Total Bids</p>
                        <p className="text-sm text-gray-900">{selectedUser.stats.totalBids}</p>
                      </div>
                      <div>
                        <p className="text-sm font-medium text-gray-500">Average Rating</p>
                        <p className="text-sm text-gray-900">
                          {selectedUser.stats.avgRating > 0 
                            ? `${selectedUser.stats.avgRating.toFixed(1)}/5 (${selectedUser.stats.totalReviews} reviews)`
                            : 'No reviews yet'
                          }
                        </p>
                      </div>
                      <div>
                        <p className="text-sm font-medium text-gray-500">Total Messages</p>
                        <p className="text-sm text-gray-900">{selectedUser.stats.totalMessages}</p>
                      </div>
                    </div>
                  </div>
                )}

                {/* Subscriptions */}
                {selectedUser.subscriptions && selectedUser.subscriptions.length > 0 && (
                  <div>
                    <h3 className="text-lg font-semibold text-gray-900 mb-3">Subscriptions</h3>
                    <div className="space-y-2">
                      {selectedUser.subscriptions.map((subscription, index) => (
                        <div key={index} className="p-3 bg-gray-50 rounded-lg">
                          <div className="flex justify-between items-center">
                            <div>
                              <p className="font-medium text-gray-900">{subscription.package.name}</p>
                              <p className="text-sm text-gray-500">
                                {subscription.package.price} {selectedUser.profile?.country?.currency || 'USD'} / {subscription.package.duration} days
                              </p>
                            </div>
                            <span className={`px-2 py-1 rounded-full text-xs font-medium ${
                              subscription.status === 'ACTIVE' ? 'bg-green-100 text-green-800' : 'bg-gray-100 text-gray-800'
                            }`}>
                              {subscription.status}
                            </span>
                          </div>
                          <p className="text-xs text-gray-500 mt-1">
                            {new Date(subscription.startDate).toLocaleDateString()} - {new Date(subscription.endDate).toLocaleDateString()}
                          </p>
                        </div>
                      ))}
                    </div>
                  </div>
                )}
              </div>
            </div>

            <div className="flex justify-end items-center px-6 py-4 border-t border-gray-200 max-md:px-4">
              <div className="flex justify-end">
                <button onClick={() => setShowUserModal(false)} className="px-4 py-2 bg-gray-500 text-white rounded-md hover:bg-gray-600">
                  Close
                </button>
              </div>
            </div>

          </div>
        </div>
      )}
    </div>
  );
}