import { NextResponse } from 'next/server';
import { getServerSession } from 'next-auth';
import { authOptions } from '@/lib/auth';
import { prisma } from '@/lib/prisma';
import { Prisma, UserRole } from '@prisma/client';

export async function GET(request: Request) {
  try {
    const session = await getServerSession(authOptions);
    
    if (!session || session.user.role !== 'ADMIN') {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
    }

    const { searchParams } = new URL(request.url);
    const page = parseInt(searchParams.get('page') || '1');
    const limit = parseInt(searchParams.get('limit') || '10');
    const role = searchParams.get('role');
    const domain = searchParams.get('domain');
    const from = searchParams.get('from');
    const to = searchParams.get('to');

    const skip = (page - 1) * limit;

    const whereCondition: Prisma.UserWhereInput = {};

    if (role) {
      whereCondition.role = role as UserRole;
    } else {
      // role not admin
      whereCondition.role = {
        not: 'ADMIN' as const
      };
    }

    if (from || to) {
      whereCondition.createdAt = {};
      if (from) whereCondition.createdAt.gte = new Date(from);
      if (to) whereCondition.createdAt.lte = new Date(to);
    }

    const profileWhere: Prisma.ProfileWhereInput = {};
    if (domain) {
      const country = await prisma.country.findFirst({
        where: { domain }
      });
      if (country) {
        profileWhere.countryId = country.id;
      }
    }

    const [users, totalCount] = await Promise.all([
      prisma.user.findMany({
        where: whereCondition,
        include: {
          profile: {
            where: Object.keys(profileWhere).length > 0 ? profileWhere : undefined,
            include: { country: true }
          },
          subscriptions: {
            where: { status: 'ACTIVE' },
            include: { package: true }
          }
        },
        orderBy: { createdAt: 'desc' },
        skip,
        take: limit
      }),
      prisma.user.count({ where: whereCondition })
    ]);

    // Filter users by domain if specified (since we can't directly filter on nested relations in where)
    const filteredUsers = domain 
      ? users.filter(user => user.profile?.country?.domain === domain)
      : users;

    const transformedUsers = filteredUsers.map(user => ({
      id: user.id,
      email: user.email,
      role: user.role,
      status: user.status,
      profile: user.profile ? {
        firstName: user.profile.firstName,
        lastName: user.profile.lastName,
        country: user.profile.country ? {
          name: user.profile.country.name,
          domain: user.profile.country.domain
        } : undefined
      } : undefined,
      subscriptions: user.subscriptions.map(sub => ({
        status: sub.status,
        package: {
          name: sub.package.name
        }
      })),
      createdAt: user.createdAt.toISOString(),
      updatedAt: user.updatedAt.toISOString()
    }));

    return NextResponse.json({
      users: transformedUsers,
      totalCount: filteredUsers.length,
      totalPages: Math.ceil(filteredUsers.length / limit),
      currentPage: page
    });
  } catch (error) {
    console.error('Error fetching users:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}

