import { NextResponse } from 'next/server';
import { getServerSession } from 'next-auth';
import { authOptions } from '@/lib/auth';
import { prisma } from '@/lib/prisma';

export async function GET() {
  try {
    const session = await getServerSession(authOptions);
    
    if (!session || session.user.role !== 'ADMIN') {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
    }

    const settings = await prisma.siteSettings.findMany();

    // Transform settings into key-value pairs
    const settingsObject: Record<string, unknown> = {};
    settings.forEach(setting => {
      let value: unknown = setting.value;
      
      // Parse JSON values
      if (setting.type === 'json') {
        try {
          value = JSON.parse(setting.value);
        } catch {
          value = setting.value;
        }
      } else if (setting.type === 'boolean') {
        value = setting.value === 'true';
      } else if (setting.type === 'number') {
        value = parseFloat(setting.value) || 0;
      }
      
      settingsObject[setting.key] = value;
    });

    return NextResponse.json(settingsObject);
  } catch (error) {
    console.error('Error fetching settings:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}

export async function PUT(request: Request) {
  try {
    const session = await getServerSession(authOptions);
    
    if (!session || session.user.role !== 'ADMIN') {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
    }

    const settingsData = await request.json();

    // Update or create settings
    const updatePromises = Object.entries(settingsData).map(async ([key, value]) => {
      let stringValue: string;
      let type: string = 'text';

      if (typeof value === 'boolean') {
        stringValue = value.toString();
        type = 'boolean';
      } else if (typeof value === 'number') {
        stringValue = value.toString();
        type = 'number';
      } else if (typeof value === 'object' && value !== null) {
        stringValue = JSON.stringify(value);
        type = 'json';
      } else {
        stringValue = String(value);
        type = 'text';
      }

      return prisma.siteSettings.upsert({
        where: { key },
        update: {
          value: stringValue,
          type,
          updatedAt: new Date()
        },
        create: {
          key,
          value: stringValue,
          type
        }
      });
    });

    await Promise.all(updatePromises);

    return NextResponse.json({ success: true });
  } catch (error) {
    console.error('Error updating settings:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}