import { NextResponse } from 'next/server';
import { getServerSession } from 'next-auth';
import { authOptions } from '@/lib/auth';
import { prisma } from '@/lib/prisma';
import { Prisma } from '@prisma/client';

export async function GET(request: Request) {
  try {
    const session = await getServerSession(authOptions);
    
    if (!session || session.user.role !== 'ADMIN') {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
    }

    const { searchParams } = new URL(request.url);
    const page = parseInt(searchParams.get('page') || '1');
    const limit = parseInt(searchParams.get('limit') || '10');
    const status = searchParams.get('status');
    const domain = searchParams.get('domain');
    const from = searchParams.get('from');
    const to = searchParams.get('to');

    const skip = (page - 1) * limit;

    const whereCondition: Prisma.ProjectWhereInput = {};

    if (status) {
      whereCondition.status = status as Prisma.EnumProjectStatusFilter;
    }

    if (domain) {
      const country = await prisma.country.findFirst({
        where: { domain }
      });
      if (country) {
        whereCondition.countryId = country.id;
      }
    }

    if (from || to) {
      whereCondition.createdAt = {};
      if (from) whereCondition.createdAt.gte = new Date(from);
      if (to) whereCondition.createdAt.lte = new Date(to);
    }

    const [projects, totalCount] = await Promise.all([
      prisma.project.findMany({
        where: whereCondition,
        include: {
          client: {
            include: { profile: true }
          },
          category: true,
          country: true,
          bids: {
            select: {
              id: true,
              status: true
            }
          }
        },
        orderBy: { createdAt: 'desc' },
        skip,
        take: limit
      }),
      prisma.project.count({ where: whereCondition })
    ]);

    const transformedProjects = projects.map(project => ({
      id: project.id,
      title: project.title,
      description: project.description,
      budget: project.budget,
      duration: project.duration,
      status: project.status,
      client: {
        email: project.client.email,
        profile: project.client.profile ? {
          firstName: project.client.profile.firstName,
          lastName: project.client.profile.lastName
        } : undefined
      },
      category: {
        name: project.category.name_en
      },
      country: {
        name: project.country.name,
        domain: project.country.domain,
        currency: project.country.currency,
        currencySymbol: project.country.currencySymbol
      },
      bids: project.bids,
      publishedAt: project.publishedAt?.toISOString(),
      completedAt: project.completedAt?.toISOString(),
      createdAt: project.createdAt.toISOString()
    }));

    return NextResponse.json({
      projects: transformedProjects,
      totalCount,
      totalPages: Math.ceil(totalCount / limit),
      currentPage: page
    });
  } catch (error) {
    console.error('Error fetching projects:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}