import { NextResponse } from 'next/server';
import { getServerSession } from 'next-auth';
import { authOptions } from '@/lib/auth';
import { prisma } from '@/lib/prisma';

export async function GET() {
  try {
    const session = await getServerSession(authOptions);
    
    if (!session || session.user.role !== 'ADMIN') {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
    }

    const packages = await prisma.package.findMany({
      include: {
        country: true,
        subscriptions: {
          select: {
            status: true
          }
        }
      },
      orderBy: [
        { country: { name: 'asc' } },
        { price: 'asc' }
      ]
    });

    const transformedPackages = packages.map(pkg => ({
      id: pkg.id,
      name: pkg.name,
      description: pkg.description,
      features: JSON.parse(pkg.features),
      price: pkg.price,
      duration: pkg.duration,
      categoryLimit: pkg.categoryLimit,
      bidLimit: pkg.bidLimit,
      isActive: pkg.isActive,
      country: {
        name: pkg.country.name,
        domain: pkg.country.domain,
        currency: pkg.country.currency,
        currencySymbol: pkg.country.currencySymbol
      },
      subscriptions: pkg.subscriptions,
      createdAt: pkg.createdAt.toISOString(),
      updatedAt: pkg.updatedAt.toISOString()
    }));

    return NextResponse.json(transformedPackages);
  } catch (error) {
    console.error('Error fetching packages:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}

export async function PUT(request: Request) {
  try {
    const session = await getServerSession(authOptions);
    
    if (!session || session.user.role !== 'ADMIN') {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
    }

    const data = await request.json();
    const { id, features, ...updateData } = data;

    const updatedPackage = await prisma.package.update({
      where: { id },
      data: {
        ...updateData,
        features: JSON.stringify(features),
        updatedAt: new Date()
      },
      include: {
        country: true,
        subscriptions: {
          select: {
            status: true
          }
        }
      }
    });

    const transformedPackage = {
      id: updatedPackage.id,
      name: updatedPackage.name,
      description: updatedPackage.description,
      features: JSON.parse(updatedPackage.features),
      price: updatedPackage.price,
      duration: updatedPackage.duration,
      categoryLimit: updatedPackage.categoryLimit,
      bidLimit: updatedPackage.bidLimit,
      isActive: updatedPackage.isActive,
      country: {
        name: updatedPackage.country.name,
        domain: updatedPackage.country.domain,
        currency: updatedPackage.country.currency,
        currencySymbol: updatedPackage.country.currencySymbol
      },
      subscriptions: updatedPackage.subscriptions,
      createdAt: updatedPackage.createdAt.toISOString(),
      updatedAt: updatedPackage.updatedAt.toISOString()
    };

    return NextResponse.json(transformedPackage);
  } catch (error) {
    console.error('Error updating package:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}