import { NextResponse } from 'next/server';
import { getServerSession } from 'next-auth';
import { authOptions } from '@/lib/auth';
import { prisma } from '@/lib/prisma';

export async function GET() {
  try {
    const session = await getServerSession(authOptions);
    
    if (!session || session.user.role !== 'ADMIN') {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
    }

    const freelancers = await prisma.user.findMany({
      where: {
        role: 'FREELANCER'
      },
      include: {
        profile: {
          include: {
            country: true
          }
        },
        receivedReviews: true,
        freelancerBids: {
          include: {
            project: true
          }
        }
      },
      orderBy: {
        createdAt: 'desc'
      }
    });

    // Calculate statistics for each freelancer
    const freelancersWithStats = freelancers.map(freelancer => {
      const reviews = freelancer.receivedReviews;
      const bids = freelancer.freelancerBids;
      
      const avgRating = reviews.length > 0 
        ? reviews.reduce((sum, review) => sum + review.rating, 0) / reviews.length 
        : 0;
      
      const totalProjects = bids.length;
      const completedProjects = bids.filter(bid => 
        bid.project.status === 'COMPLETED'
      ).length;

      // Transform skills from JSON string to array
      let skills = [];
      if (freelancer.profile?.skills) {
        try {
          skills = JSON.parse(freelancer.profile.skills);
        } catch {
          skills = [];
        }
      }

      return {
        id: freelancer.id,
        email: freelancer.email,
        profile: {
          firstName: freelancer.profile?.firstName,
          lastName: freelancer.profile?.lastName,
          title: freelancer.profile?.title,
          hourlyRate: freelancer.profile?.hourlyRate,
          skills: skills,
          profileImage: freelancer.profile?.profileImage,
          country: freelancer.profile?.country
        },
        avgRating: Math.round(avgRating * 10) / 10,
        totalProjects,
        completedProjects,
        status: freelancer.status,
        isVerified: freelancer.emailVerified !== null,
        createdAt: freelancer.createdAt.toISOString()
      };
    });

    return NextResponse.json(freelancersWithStats);
  } catch (error) {
    console.error('Error fetching freelancers:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}