import { NextRequest, NextResponse } from "next/server";
import { prisma } from "@/lib/prisma";

export async function GET(req: NextRequest) {
  try {
    const { searchParams } = new URL(req.url);
    const userId = searchParams.get("userId"); // user ID passed as query param
    if (!userId) {
      return NextResponse.json(
        {
          success: false,
          error: "Missing userId",
          message: "Please provide a valid user ID",
        },
        { status: 400 }
      );
    }

    const page = parseInt(searchParams.get("page") || "1");
    // const limit = parseInt(searchParams.get("limit") || "10");
    const limit = 100000;
    const statusFilter = searchParams.get("status") || "all";
    const dateRange = searchParams.get("dateRange");

    // let whereClause: any = { userId };

    // Apply status filter
    // if (statusFilter && statusFilter !== "all") {
    //   const statusMap: { [key: string]: string } = {
    //     active: "ACTIVE",
    //     expired: "EXPIRED",
    //     cancelled: "CANCELLED",
    //   };
    //   whereClause.status = statusMap[statusFilter] || "ACTIVE";
    // }

    // Apply date range filter
    // if (dateRange) {
    //   const [startDate, endDate] = dateRange.split(",");
    //   if (startDate && endDate) {
    //     whereClause.createdAt = {
    //       gte: new Date(startDate),
    //       lte: new Date(endDate),
    //     };
    //   }
    // }

    const totalCount = await prisma.categorySubscription.count({
      where: {
          userId: userId, // or simply `userId` (shorthand)
        },
    });

    const subscriptions = await prisma.categorySubscription.findMany({
      where: {userId: userId},
      include: { category: true },
      orderBy: { createdAt: "desc" },
      skip: (page - 1) * limit,
      take: limit,
    });

    const transactions = subscriptions.map((subscription: any) => {
      const now = new Date();
      const endDate = new Date(subscription.endDate);
      const diffTime = endDate.getTime() - now.getTime();
      const expiryDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));

      let status: "active" | "expired" | "expiring" = "expired";
      if (subscription.status === "ACTIVE") {
        if (expiryDays > 7) status = "active";
        else if (expiryDays > 0) status = "expiring";
      }

      const transactionId = `TXN${subscription.createdAt.getFullYear()}${String(
        subscription.createdAt.getMonth() + 1
      ).padStart(2, "0")}${String(subscription.createdAt.getDate()).padStart(
        2,
        "0"
      )}${subscription.id.slice(-6)}`;

      return {
        id: subscription.id,
        transactionId,
        category: subscription.category.name,
        categoryIcon:
          subscription.category.icon ||
          `/assets/image/category/${subscription.category.slug}.svg`,
        amount: 149.9,
        currency: "EUR",
        paymentMethod: "card",
        status,
        paymentStatus: "completed",
        date: subscription.createdAt.toISOString(),
        startDate: subscription.startDate.toISOString(),
        endDate: subscription.endDate.toISOString(),
      };
    });

    const totalPages = Math.ceil(totalCount / limit);
    const hasNext = page < totalPages;
    const hasPrev = page > 1;

    return NextResponse.json({
      success: true,
      message: `Found ${totalCount} membership record${
        totalCount !== 1 ? "s" : ""
      }`,
      data: {
        transactions,
        pagination: {
          currentPage: page,
          totalPages,
          totalItems: totalCount,
          itemsPerPage: limit,
          hasNextPage: hasNext,
          hasPrevPage: hasPrev,
        },
      },
    });
  } catch (error) {
    console.error("Error fetching membership history:", error);
    return NextResponse.json(
      {
        success: false,
        error: "Failed to load membership history",
        message:
          "We encountered an issue loading membership transactions. Please try again.",
        details:
          error instanceof Error ? error.message : "Unknown server error",
        type: "server_error",
      },
      { status: 500 }
    );
  }
}

// export async function GET(req: NextRequest) {
//   try {
//     const { searchParams } = new URL(req.url);
//     const userId = searchParams.get('userId');  
//     if (!userId) {
//       return NextResponse.json(
//         { success: false, error: 'Missing userId', message: 'Please provide a valid user ID' },
//         { status: 400 }
//       );
//     }

//     const statusFilter = searchParams.get('status') || 'all';
//     const dateRange = searchParams.get('dateRange');

//     // let whereClause: any = { userId };
//     const whereClause = { userId } as any;

//     // Apply status filter
//     if (statusFilter && statusFilter !== 'all') {
//       const statusMap: { [key: string]: string } = {
//         active: 'ACTIVE',
//         expired: 'EXPIRED',
//         cancelled: 'CANCELLED'
//       };
//       whereClause.status = statusMap[statusFilter] || 'ACTIVE';
//     }

//     // Apply date range filter
//     if (dateRange) {
//       const [startDate, endDate] = dateRange.split(',');
//       if (startDate && endDate) {
//         whereClause.createdAt = {
//           gte: new Date(startDate),
//           lte: new Date(endDate)
//         };
//       }
//     }

//     const subscriptions = await prisma.categorySubscription.findMany({
//       where: whereClause,
//       include: { category: true },
//       orderBy: { createdAt: 'desc' },
//     });

//     const transactions = subscriptions.map((subscription: any) => {
//       const now = new Date();
//       const endDate = new Date(subscription.endDate);
//       const diffTime = endDate.getTime() - now.getTime();
//       const expiryDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));

//       let status: 'active' | 'expired' | 'expiring' = 'expired';
//       if (subscription.status === 'ACTIVE') {
//         if (expiryDays > 7) status = 'active';
//         else if (expiryDays > 0) status = 'expiring';
//       }

//       const transactionId = `TXN${subscription.createdAt.getFullYear()}${String(
//         subscription.createdAt.getMonth() + 1
//       ).padStart(2, '0')}${String(subscription.createdAt.getDate()).padStart(2, '0')}${subscription.id.slice(-6)}`;

//       return {
//         id: subscription.id,
//         transactionId,
//         category: subscription.category.name,
//         categoryIcon: subscription.category.icon || `/assets/image/category/${subscription.category.slug}.svg`,
//         amount: 149.9,
//         currency: 'EUR',
//         paymentMethod: 'card',
//         status,
//         paymentStatus: 'completed',
//         date: subscription.createdAt.toISOString(),
//         startDate: subscription.startDate.toISOString(),
//         endDate: subscription.endDate.toISOString()
//       };
//     });

//     return NextResponse.json({
//       success: true,
//       message: `Found ${subscriptions.length} membership record${subscriptions.length !== 1 ? 's' : ''}`,
//       data: { transactions }
//     });
//   } catch (error) {
//     console.error('Error fetching membership history:', error);
//     return NextResponse.json(
//       {
//         success: false,
//         error: 'Failed to load membership history',
//         message: 'We encountered an issue loading membership transactions. Please try again.',
//         details: error instanceof Error ? error.message : 'Unknown server error',
//         type: 'server_error'
//       },
//       { status: 500 }
//     );
//   }
// }
