import { NextResponse } from 'next/server';
import { getServerSession } from 'next-auth';
import { authOptions } from '@/lib/auth';
import { prisma } from '@/lib/prisma';

export async function GET() {
  try {
    const session = await getServerSession(authOptions);
    
    if (!session || session.user.role !== 'ADMIN') {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
    }

    // Get total counts
    const [totalUsers, totalProjects, totalFreelancers, activeSubscriptions] = await Promise.all([
      prisma.user.count({ where: { role: { not: 'ADMIN' } } }),
      prisma.project.count(),
      prisma.user.count({ where: { role: 'FREELANCER' } }),
      prisma.subscription.count({ where: { status: 'ACTIVE' } })
    ]);

    // Calculate total revenue from active subscriptions
    const activeSubscriptionsWithPackages = await prisma.subscription.findMany({
      where: { status: 'ACTIVE' },
      include: { package: true }
    });

    const totalRevenue = activeSubscriptionsWithPackages.reduce(
      (sum, sub) => sum + sub.package.price, 0
    );

    // Calculate growth from last month
    const lastMonth = new Date();
    lastMonth.setMonth(lastMonth.getMonth() - 1);

    const [usersLastMonth, projectsLastMonth, freelancersLastMonth] = await Promise.all([
      prisma.user.count({ where: { createdAt: { gte: lastMonth }, role: { not: 'ADMIN' } } }),
      prisma.project.count({ where: { createdAt: { gte: lastMonth } } }),
      prisma.user.count({ 
        where: { 
          role: 'FREELANCER',
          createdAt: { gte: lastMonth }
        }
      })
    ]);

    const userGrowth = totalUsers > 0 ? ((usersLastMonth / totalUsers) * 100) : 0;
    const projectGrowth = totalProjects > 0 ? ((projectsLastMonth / totalProjects) * 100) : 0;
    const freelancerGrowth = totalFreelancers > 0 ? ((freelancersLastMonth / totalFreelancers) * 100) : 0;

    return NextResponse.json({
      totalUsers,
      totalProjects,
      totalFreelancers,
      totalRevenue,
      activeSubscriptions,
      monthlyGrowth: {
        users: Math.round(userGrowth * 10) / 10,
        projects: Math.round(projectGrowth * 10) / 10,
        freelancers: Math.round(freelancerGrowth * 10) / 10
      }
    });
  } catch (error) {
    console.error('Error fetching dashboard stats:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}