import { NextResponse } from 'next/server';
import { getServerSession } from 'next-auth';
import { authOptions } from '@/lib/auth';
import { prisma } from '@/lib/prisma';

export async function GET(request: Request) {
  try {
    const session = await getServerSession(authOptions);
    
    if (!session || session.user.role !== 'ADMIN') {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
    }

    const { searchParams } = new URL(request.url);
    const from = searchParams.get('from');
    const to = searchParams.get('to');
    const domain = searchParams.get('domain');

    const dateFilter: { gte?: Date; lte?: Date } = {};
    if (from) dateFilter.gte = new Date(from);
    if (to) dateFilter.lte = new Date(to);

    const whereCondition: { createdAt?: { gte?: Date; lte?: Date } } = {};
    if (Object.keys(dateFilter).length > 0) {
      whereCondition.createdAt = dateFilter;
    }

    // Get monthly data for the last 6 months
    const sixMonthsAgo = new Date();
    sixMonthsAgo.setMonth(sixMonthsAgo.getMonth() - 6);

    const monthlyData = [];
    const monthNames = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];

    for (let i = 5; i >= 0; i--) {
      const startDate = new Date();
      startDate.setMonth(startDate.getMonth() - i);
      startDate.setDate(1);
      startDate.setHours(0, 0, 0, 0);

      const endDate = new Date(startDate);
      endDate.setMonth(endDate.getMonth() + 1);
      endDate.setDate(0);
      endDate.setHours(23, 59, 59, 999);

      const userMonthWhere = {
        role: {
          not: 'ADMIN' as const
        },
        createdAt: {
          gte: startDate,
          lte: endDate
        }
      };

      const projectMonthWhere = {
        createdAt: {
          gte: startDate,
          lte: endDate
        },
        client: {
          role: {
            not: 'ADMIN' as const
          }
        }
      };

      if (domain) {
        const countryCondition = await prisma.country.findFirst({
          where: { domain }
        });
        if (countryCondition) {
          Object.assign(userMonthWhere, {
            profile: {
              countryId: countryCondition.id
            }
          });
          Object.assign(projectMonthWhere, {
            countryId: countryCondition.id
          });
        }
      }

      const [users, projects] = await Promise.all([
        prisma.user.count({ where: userMonthWhere }),
        prisma.project.count({ where: projectMonthWhere })
      ]);

      // Calculate revenue for this month
      const subscriptionsThisMonth = await prisma.subscription.findMany({
        where: {
          createdAt: {
            gte: startDate,
            lte: endDate
          }
        },
        include: { package: true }
      });

      const revenue = subscriptionsThisMonth.reduce(
        (sum, sub) => sum + sub.package.price, 0
      );

      monthlyData.push({
        name: monthNames[startDate.getMonth()],
        users,
        projects,
        revenue: Math.round(revenue)
      });
    }

    // Generate revenue trend data
    const revenueData = monthlyData.map(month => ({
      name: month.name,
      revenue: month.revenue
    }));

    return NextResponse.json({
      monthlyData,
      revenueData
    });
  } catch (error) {
    console.error('Error fetching dashboard charts:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}