import { NextResponse } from 'next/server';
import { getServerSession } from 'next-auth';
import { authOptions } from '@/lib/auth';
import { prisma } from '@/lib/prisma';

export async function GET() {
  try {
    const session = await getServerSession(authOptions);
    
    if (!session || session.user.role !== 'ADMIN') {
      return NextResponse.json({ error: 'Unauthorized' }, { status: 401 });
    }

    const activities: Array<{
      action: string;
      user: string;
      time: string;
      createdAt: Date;
    }> = [];

    // Get recent user registrations
    const recentUsers = await prisma.user.findMany({
      orderBy: { createdAt: 'desc' },
      take: 5,
      include: { profile: true }
    });

    recentUsers.forEach(user => {
      const timeDiff = Date.now() - new Date(user.createdAt).getTime();
      const timeAgo = formatTimeAgo(timeDiff);
      
      activities.push({
        action: 'New user registered',
        user: user.email,
        time: timeAgo,
        createdAt: user.createdAt
      });
    });

    // Get recently completed projects
    const completedProjects = await prisma.project.findMany({
      where: { status: 'COMPLETED' },
      orderBy: { completedAt: 'desc' },
      take: 3,
      include: {
        client: {
          include: { profile: true }
        }
      }
    });

    completedProjects.forEach(project => {
      if (project.completedAt) {
        const timeDiff = Date.now() - new Date(project.completedAt).getTime();
        const timeAgo = formatTimeAgo(timeDiff);
        
        activities.push({
          action: 'Project completed',
          user: project.title,
          time: timeAgo,
          createdAt: project.completedAt
        });
      }
    });

    // Get recent subscriptions
    const recentSubscriptions = await prisma.subscription.findMany({
      where: { status: 'ACTIVE' },
      orderBy: { createdAt: 'desc' },
      take: 3,
      include: { 
        package: true,
        user: { include: { profile: true } }
      }
    });

    recentSubscriptions.forEach(subscription => {
      const timeDiff = Date.now() - new Date(subscription.createdAt).getTime();
      const timeAgo = formatTimeAgo(timeDiff);
      
      activities.push({
        action: 'New subscription',
        user: subscription.package.name + ' Plan',
        time: timeAgo,
        createdAt: subscription.createdAt
      });
    });

    // Get recently verified freelancers
    const verifiedFreelancers = await prisma.user.findMany({
      where: { 
        role: 'FREELANCER',
        emailVerified: { not: null }
      },
      orderBy: { emailVerified: 'desc' },
      take: 3,
      include: { profile: true }
    });

    verifiedFreelancers.forEach(freelancer => {
      if (freelancer.emailVerified) {
        const timeDiff = Date.now() - new Date(freelancer.emailVerified).getTime();
        const timeAgo = formatTimeAgo(timeDiff);
        
        activities.push({
          action: 'Freelancer verified',
          user: freelancer.email,
          time: timeAgo,
          createdAt: freelancer.emailVerified
        });
      }
    });

    // Sort activities by creation time and take the most recent 10
    const sortedActivities = activities
      .sort((a, b) => new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime())
      .slice(0, 10)
      .map(activity => ({
        action: activity.action,
        user: activity.user,
        time: activity.time
      }));

    return NextResponse.json(sortedActivities);
  } catch (error) {
    console.error('Error fetching dashboard activities:', error);
    return NextResponse.json(
      { error: 'Internal server error' },
      { status: 500 }
    );
  }
}

function formatTimeAgo(milliseconds: number): string {
  const seconds = Math.floor(milliseconds / 1000);
  const minutes = Math.floor(seconds / 60);
  const hours = Math.floor(minutes / 60);
  const days = Math.floor(hours / 24);

  if (days > 0) return `${days} day${days > 1 ? 's' : ''} ago`;
  if (hours > 0) return `${hours} hour${hours > 1 ? 's' : ''} ago`;
  if (minutes > 0) return `${minutes} minute${minutes > 1 ? 's' : ''} ago`;
  return `${seconds} second${seconds > 1 ? 's' : ''} ago`;
}