import { NextResponse } from "next/server";
import { getServerSession } from "next-auth";
import { authOptions } from "@/lib/auth";
import { prisma } from "@/lib/prisma";

export async function GET() {
  try {
    const session = await getServerSession(authOptions);

    if (!session || session.user.role !== "ADMIN") {
      return NextResponse.json({ error: "Unauthorized" }, { status: 401 });
    }

    const countries = await prisma.country.findMany({
      orderBy: { name: "asc" },
      include: {
        packages: true, // ✅ corrected relation name
        profile: true,
        project: true,
      },
    });

    const transformedCountries = countries.map((country: any) => ({
      id: country.id,
      name: country.name,
      code: country.code,
      domain: country.domain,
      languages: JSON.parse(country.languages || "[]"),
      defaultLanguage: country.defaultLanguage,
      currency: country.currency,
      currencySymbol: country.currencySymbol,
      packagePrice: country.packagePrice,
      paymentGateways: JSON.parse(country.paymentGateways || "[]"),
      isActive: country.isActive,
      createdAt: country.createdAt.toISOString(),
      updatedAt: country.updatedAt.toISOString(),
      customerPhone:country?.customerPhone,
      // Optional fields
      businessEmail: country.businessEmail,
      facebookUrl: country.facebookUrl,
      mapIframe: country.mapIframe,
      googleUrl: country.googleUrl,
      instagramUrl: country.instagramUrl,
      linkedinUrl: country.linkedinUrl,
      logoFooter: country.logoFooter,
      logoHeader: country.logoHeader,
      supportEmail: country.supportEmail,
      twitterUrl: country.twitterUrl,

      location_en: country?.location_en,
      location_fr: country?.location_fr,
      location_de: country?.location_de,
      location_it: country?.location_it,
      location_pt: country?.location_pt,
      location_es: country?.location_es,

      sublocation_en: country?.sublocation_en,
      sublocation_fr: country?.sublocation_fr,
      sublocation_de: country?.sublocation_de,
      sublocation_it: country?.sublocation_it,
      sublocation_pt: country?.sublocation_pt,
      sublocation_es: country?.sublocation_es,

      shortDescription_en: country?.shortDescription_en,
      shortDescription_fr: country?.shortDescription_fr,
      shortDescription_de: country?.shortDescription_de,
      shortDescription_it: country?.shortDescription_it,
      shortDescription_pt: country?.shortDescription_pt,
      shortDescription_es: country?.shortDescription_es,
      // Related models
      packages: country.packages, // ✅ renamed correctly
      profile: country.profile,
      project: country.project,
    }));

    return NextResponse.json(transformedCountries);
  } catch (error) {
    console.error("Error fetching countries:", error);
    return NextResponse.json(
      { error: "Internal server error" },
      { status: 500 }
    );
  }
}
